#include "builcule_draw.h"
#include <algorithm>

Builcule_Draw::Builcule_Draw(QWidget *parent)
 : Widget_Draw_Base(parent),
   UnitPtr(nullptr),
   AffineTrans(Affine_Trans::DEFAULT)
{}

void Builcule_Draw::calc_eye_pos(
        const Eigen::Vector3d *center,
        double radius) {
 DrawCenter = *center;
 DrawRadius = radius + 2.5;

 EyePos[0] = DrawCenter[0];
 EyePos[1] = DrawCenter[1];
 EyePos[2] = DrawCenter[2] + DrawRadius * 10.0;

 Front = DrawRadius * 9.0;
 Back = DrawRadius * 11.0;

 if(Front < 10.0)
  Slices = 64;
 else if(Front < 100.0)
  Slices = 32;
 else
  Slices = 16;

 Stacks = Slices / 2;
}

void Builcule_Draw::push_wire() {
 const auto *atom_vect = UnitPtr->get_atom_vect();
 Wire wire;
 for(const auto &atom0 : *atom_vect) {
  wire.Pos0 = atom0->get_xyz();
  auto bond = atom0->get_bond_up();
  for(const auto &atom1 : *bond) {
   wire.Pos1 = atom1->get_xyz();
   WireVect.push_back(wire);
  }
 }
}

void Builcule_Draw::push_cylinder() {
 const auto *atom_vect = UnitPtr->get_atom_vect();
 Cylinder cylinder;
 cylinder.Radius = StickRadius;
 for(const auto &atom0 : *atom_vect) {
  cylinder.Pos0 = atom0->get_xyz();
  auto bond = atom0->get_bond_up();
  for(const auto &atom1 : *bond) {
   cylinder.Pos1 = atom1->get_xyz();
   CylinderVect.push_back(cylinder);
  }
 }
}

void Builcule_Draw::push_checked_cylinder() {
 const auto *atom_vect = UnitPtr->get_atom_vect();
 Cylinder cylinder;
 cylinder.Radius = StickRadius;
 for(const auto &atom0 : *atom_vect) {
  if(atom0->get_check_num()) {
   cylinder.Pos0 = atom0->get_xyz();
   auto bond = atom0->get_bond_up();
   for(const auto &atom1 : *bond) {
    if(atom1->get_check_num()){
     cylinder.Pos1 = atom1->get_xyz();
     CylinderVect.push_back(cylinder);
    }
   }
  }
 }
}

void Builcule_Draw::push_sphere() {
 SerialToSphereIdx.clear();
 SphereVect.clear();
 SphereVect.shrink_to_fit();

 const auto *atom_vect = UnitPtr->get_atom_vect();
 int atom_cnt = atom_vect->size();
 Sphere sphere;
 for(int i = 0; i != atom_cnt; ++i) {
  const auto &atom = (*atom_vect)[i];
  sphere.Pos = atom->get_xyz();
  sphere.SerialNum = atom->get_serial_num();
  sphere.Elmt = atom->get_elmt();
  sphere.Symbol = QString(atom->get_symbol()->data());
  sphere.MolIdx = atom->get_mol_num();
  sphere.PeptIdx = atom->get_pept_num();
  sphere.SeqIdx = atom->get_seq_num();
  sphere.Code = *SharedData.get_aa_code_3(atom->get_code());
  sphere.Annot = AnnotMap[atom->get_aa_atom()];
  SphereVect.push_back(sphere);

  set_sphere_color_cpk(i);
  SerialToSphereIdx[sphere.SerialNum] = i;
 }

 //ピックされた原子の色を変更
 for(int serial : PickedSerial) {
  int idx = SerialToSphereIdx[serial];
  set_sphere_color_pick(idx);
 }
}

void Builcule_Draw::set_sphere_radius(Draw_Model model) {
 int cnt = SphereVect.size();
 if(model == Draw_Model::BallStick) {
  for(int i = 0; i != cnt; ++i)
   set_sphere_radius_ball(i);
 }
 else if(model == Draw_Model::Spacefill) {
  for(int i = 0; i != cnt; ++i)
   set_sphere_radius_spacefill(i);
 }
 else
  std::cerr << __PRETTY_FUNCTION__ << std::endl;
}

void Builcule_Draw::push_checked_sphere() {
 SerialToSphereIdx.clear();
 SphereVect.clear();
 SphereVect.shrink_to_fit();

 const auto *atom_vect = UnitPtr->get_atom_vect();
 Sphere sphere;
 int cnt = 0;
 for(const auto &atom : *atom_vect) {
  if(atom->get_check_num()) {
   sphere.Pos = atom->get_xyz();
   sphere.SerialNum = atom->get_serial_num();
   sphere.Elmt = atom->get_elmt();
   sphere.Symbol = QString(atom->get_symbol()->data());
   sphere.MolIdx = atom->get_mol_num();
   sphere.PeptIdx = atom->get_pept_num();
   sphere.SeqIdx = atom->get_seq_num();
   sphere.Code = *SharedData.get_aa_code_3(atom->get_code());
   sphere.Annot = AnnotMap[atom->get_aa_atom()];
   SphereVect.push_back(sphere);

   //ピックされた原子の色は変更
   if(PickedSerial.end() != std::find(
        PickedSerial.begin(), PickedSerial.end(), sphere.SerialNum))
    set_sphere_color_pick(cnt);
   else
    set_sphere_color_cpk(cnt);

   SerialToSphereIdx[sphere.SerialNum] = cnt++;
  }
 }
}

void Builcule_Draw::set_checked_sphere_radius(Draw_Model model) {
 const auto *atom_vect = UnitPtr->get_atom_vect();
 int cnt = 0;
 if(model == Draw_Model::BallStick) {
  for(const auto &atom : *atom_vect) {
   if(atom->get_check_num()) {
    set_sphere_radius_ball(cnt++);
   }
  }
 }

 else if(model == Draw_Model::Spacefill) {
  for(const auto &atom : *atom_vect) {
   if(atom->get_check_num()) {
    set_sphere_radius_spacefill(cnt++);
   }
  }
 }
}

void Builcule_Draw::set_checked_sphere_radius() {
 const auto *atom_vect = UnitPtr->get_atom_vect();
 int cnt = 0;
 for(const auto &atom : *atom_vect) {
  int lv = atom->get_check_num();
  if(lv > 1)
   set_sphere_radius_spacefill(cnt++);
  else if(lv > 0)
   set_sphere_radius_ball(cnt++);
 }
}

void Builcule_Draw::set_sphere_radius_ball(int idx) {
 if(SphereVect[idx].Elmt != 1)
  SphereVect[idx].Radius = BallRadiusX;
 else
  SphereVect[idx].Radius = BallRadiusH;
}

void Builcule_Draw::set_sphere_radius_spacefill(int idx) {
 int elmt = SphereVect[idx].Elmt;
 SphereVect[idx].Radius = SharedData.get_vdw_radius(elmt);
}

void Builcule_Draw::set_sphere_color_cpk(int idx) {
 for(int i = 0; i != 4; ++i)
  SphereVect[idx].Color[i] = ColorCPK[SphereVect[idx].Elmt][i];
}

void Builcule_Draw::set_sphere_color_pick(int idx) {
 for(int i = 0; i != 4; ++i)
  SphereVect[idx].Color[i] = ColorCPK[SharedData.ElmtCnt + 1][i];
}

void Builcule_Draw::clear_obj() {
 //選択分子の描画等で必要となる場合があるので，PickedSerial はクリアしない
 //ファイルオープン等の場合は，clear_picked() も呼び出す
 SphereVect.clear();
 SphereVect.shrink_to_fit();
 CylinderVect.clear();
 CylinderVect.shrink_to_fit();
 WireVect.clear();
 WireVect.shrink_to_fit();
 SerialToSphereIdx.clear();
}

void Builcule_Draw::clear_picked_obj() {
 for(int serial : PickedSerial) {
  int idx = SerialToSphereIdx[serial];
  set_sphere_color_cpk(idx);
 }

 PickedSerial.clear();
 PickedSerial.shrink_to_fit();
}

void Builcule_Draw::paintGL() {
 glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

 set_projection();
 set_lookat();

 gluQuadricDrawStyle(QSurface, GLU_FILL);
 glEnable(GL_LIGHTING);
 glEnable(GL_LIGHT0);
 glEnable(GL_CULL_FACE);
 glCullFace(GL_BACK);

 if(!SphereVect.empty())
  draw_sphere();
 if(!CylinderVect.empty())
  draw_cylinder();

 //gluQuadricDrawStyle(QSurface, GLU_LINE);  //不要だ
 glDisable(GL_LIGHTING);
 glDisable(GL_CULL_FACE);

 if(!WireVect.empty())
  draw_wire();
}

void Builcule_Draw::draw_wire() {
 glBegin(GL_LINES);
 for(const Wire &wire : WireVect) {
  glColor3f(wire.Color[0], wire.Color[1], wire.Color[2]);
  glVertex3f((*wire.Pos0)[0], (*wire.Pos0)[1], (*wire.Pos0)[2]);
  glVertex3f((*wire.Pos1)[0], (*wire.Pos1)[1], (*wire.Pos1)[2]);
 }
 glEnd();
}

void Builcule_Draw::draw_sphere() {
 if(SphereVect.empty()) return;

 for(const Sphere &sphere : SphereVect) {
  glMaterialfv(GL_FRONT, GL_DIFFUSE, sphere.Color);
  glPushMatrix();
  glTranslated((*sphere.Pos)[0],
        (*sphere.Pos)[1],
        (*sphere.Pos)[2]);
  gluSphere(QSurface, sphere.Radius, Slices, Stacks);
  glPopMatrix();
 }
}

void Builcule_Draw::draw_cylinder() {
 Eigen::Vector3d tmp_vect{0.0, 0.0, 0.0};
 Eigen::Vector3d tmp_axis{0.0, 0.0, 0.0};
 for(const Cylinder &cylinder : CylinderVect) {
  tmp_vect = (*cylinder.Pos1) - (*cylinder.Pos0);
  double norm = tmp_vect.norm();
  tmp_vect = tmp_vect.normalized();
  //結合と Z 軸との成す角を内積 (z 軸 = 0.0, 0.0, 1.0) から求める
  double angle = acos(tmp_vect[2]) * RADtoDEG;
  //回転軸を外積から求める
  tmp_axis = ZAxis.cross(tmp_vect).normalized();
  const Eigen::Vector3d &pos = (*cylinder.Pos0);
  glMaterialfv(GL_FRONT, GL_DIFFUSE, cylinder.Color);
  glPushMatrix();
  glTranslated(pos[0], pos[1], pos[2]);  //シリンダを平行移動
  glRotated(angle, tmp_axis[0], tmp_axis[1], tmp_axis[2]);  //同回転
  gluCylinder(
        QSurface,
        cylinder.Radius,
        cylinder.Radius,
        norm,
        Slices,
        1);
   glPopMatrix ();
 }
}

bool Builcule_Draw::obj_empty() {
 if(!SphereVect.empty()) return false;
 if(!CylinderVect.empty()) return false;
 if(!WireVect.empty()) return false;
 return true;
}

void Builcule_Draw::mousePressEvent(QMouseEvent *event) {
 if(obj_empty()) return;

 PressedButton = event->button();
 if(AffineTrans == Affine_Trans::EYE_XY) {
  if(PressedButton == Qt::RightButton)
   AffineTrans = Affine_Trans::DEFAULT;
 }
 else if(AffineTrans == Affine_Trans::MOL) {
  if(PressedButton == Qt::LeftButton)
   AffineTrans = Affine_Trans::MOL_ROTATE;
  else if(PressedButton == Qt::RightButton)
   AffineTrans = Affine_Trans::MOL_XY;
  else
   AffineTrans = Affine_Trans::DEFAULT;
 }
 else if(AffineTrans == Affine_Trans::REGION_MOVE) {
  if(PressedButton != Qt::RightButton)
   AffineTrans = Affine_Trans::DEFAULT;
 }
 else if(AffineTrans == Affine_Trans::REGION_ROTATE) {
  if(PressedButton != Qt::LeftButton)
   AffineTrans = Affine_Trans::DEFAULT;
 }

 //回転用の設定．描画領域は正方形に設定されている
 MouseRotateRatio = 180.0 / (double)width();
 MouseMoveRatio = 2.0 * DrawRadius / (double)width();
 MouseX = event->x();
 MouseY = event->y();

 int serial = pick();
 if(serial < 0)  //ピックされなかった場合
  return;
 int idx = SerialToSphereIdx[serial];
 if(SphereVect[idx].Color[0]
        != ColorCPK[SharedData.ElmtCnt + 1][0]) {
  PickedSerial.push_back(serial);
  set_sphere_color_pick(idx);
  repaint();
 }
 else {
  delete_picked_serial(serial);
  set_sphere_color_cpk(idx);
  repaint();
 }

 //ピックされた原子の情報を表示
 const Sphere &sphere = SphereVect[idx];
 QString info("元素：" + sphere.Symbol);
 info += QString("\n分子番号：")
       += QString::number(sphere.MolIdx);
 if(sphere.PeptIdx > -1) {
  info += QString("\nペプチド番号：")
       += QString::number(sphere.PeptIdx)
       += QString("\n配列番号：")
       += QString::number(sphere.SeqIdx)
       += QString("\nアミノ酸コード：")
       += QString(sphere.Code.data())
       += QString("\n表示記号：")
       += QString(sphere.Annot.data());
 }

//  デバッグ用
 info += QString("\nシリアル番号：")
      += QString::number(sphere.SerialNum);

 PopupLabel->setText(info);
 PopupWidget->move(event->globalX() + 10, event->globalY() + 10);
 PopupWidget->show();
}

void Builcule_Draw::delete_picked_serial(int serial) {
 int cnt = PickedSerial.size();
 for(int i = 0; i < cnt; ++i) {
  if(PickedSerial[i] == serial) {
   PickedSerial.erase(PickedSerial.begin() + i);
   return;
  }
 }
}

void Builcule_Draw::mouseMoveEvent(QMouseEvent *event) {
 if(obj_empty()) return;

 if(AffineTrans == Affine_Trans::ZOOM)
  AffineTrans = Affine_Trans::DEFAULT;

 //MouseX，MouseY と event->x(), event->y() との差が移動度
 if(PressedButton == Qt::LeftButton) {
  //ユニットを回転
  if(AffineTrans == Affine_Trans::DEFAULT)
   rotate_unit(event->x(), event->y());
  //第一選択分子をその場で回転
  else if(AffineTrans == Affine_Trans::MOL_ROTATE)
   rotate_mol(event->x(), event->y());
   //結合角または二面角を変更
  else if(AffineTrans == Affine_Trans::REGION_ROTATE)
   rotate_region(event->x());
 }
 else if(PressedButton == Qt::RightButton) {
  //視点を移動
  if(AffineTrans == Affine_Trans::EYE_XY)
   move_eye(event->x(), event->y());
  //選択分子を平行移動
  else if(AffineTrans == Affine_Trans::MOL_XY)
   move_mol(event->x(), event->y());
  //結合長を変更
  else if(AffineTrans == Affine_Trans::REGION_MOVE)
   move_region(event->x());
 }

 //MouseX，MouseY を再設定
 MouseX = event->x();
 MouseY = event->y();

 if(PopupWidget->isVisible())
  PopupWidget->move(event->globalX() + 10, event->globalY() + 10);

 repaint();
}

void Builcule_Draw::rotate_unit(int x0, int y0) {
 double dx = (double)(x0 - MouseX) * MouseRotateRatio;
 double dy = (double)(y0 - MouseY) * MouseRotateRatio;
 UnitPtr->rotate_unit(dx, dy);
}

void Builcule_Draw::rotate_mol(int x0, int y0) {
 double dx = (double)(x0 - MouseX) * MouseRotateRatio;
 double dy = (double)(y0 - MouseY) * MouseRotateRatio;
 UnitPtr->rotate_mol(dx, dy);
}

void Builcule_Draw::move_mol(int x0, int y0) {
 double dx = (double)(x0 - MouseX);
 double dy = (double)(MouseY - y0);
 double dx_length = dx * MouseMoveRatio;
 double dy_length = dy * MouseMoveRatio;
 UnitPtr->move_mol(dx_length, dy_length);
}

void Builcule_Draw::rotate_region(int x0) {
 double dx = (double)(x0 - MouseX) * MouseRotateRatio;
 UnitPtr->rotate_region(dx * DEGtoRAD);
}

void Builcule_Draw::move_region(int x0) {
 double dx = (double)(x0 - MouseX);
 double dx_length = dx * MouseMoveRatio;
 UnitPtr->move_region(dx_length);
}

void Builcule_Draw::mouseReleaseEvent(QMouseEvent *event) {
 if(obj_empty()) return;

 MouseX = event->x();
 MouseY = event->y();
 AffineTrans = Affine_Trans::DEFAULT;
 PressedButton = Qt::NoButton;

 UnitPtr->init_rotation_center();

 if(PopupWidget->isVisible())
  PopupWidget->hide();
}

void Builcule_Draw::wheelEvent(QWheelEvent *event) {
 if(AffineTrans != Affine_Trans::ZOOM)
  return;

 if(obj_empty()) return;

 double d = event->angleDelta().y();
 d *= DrawRadius;
 d /= 500.0;

 if(EyePos[2] >= 0.0)
  EyePos[2] -= d;
 else
  EyePos[2] += d;

 Front -= d;
 Back -= d;

 repaint();
}

void Builcule_Draw::move_eye(int x0, int y0) {
 double dx =
        (double)(MouseX - x0) / (double)width() * DrawRadius;
 double dy
        = (double)(MouseY - y0) / (double)height() * DrawRadius;

 DrawCenter[0] += dx;
 DrawCenter[1] -= dy;
 EyePos[0] += dx;
 EyePos[1] -= dy;
}

int Builcule_Draw::pick() {
 //ビューポートを設定する箇所はここだけでよい
 glViewport(0, 0, width(), height());

 //クリック位置に描画されている原子を格納するための配列を確保
 const GLuint selBufferSize(400);  //1 原子につき配列 4 個分を利用する．ここでは 100 原子分
 GLuint *selection = new GLuint[selBufferSize];
 glSelectBuffer(selBufferSize, selection);
 GLint selection_viewport[4];
 glGetIntegerv(GL_VIEWPORT, selection_viewport);

 //ここは，gluPickMatrix() の呼び出しがあるので，set_projection(); に書き換えられない．保守に注意
 glMatrixMode(GL_PROJECTION);
 glLoadIdentity();
 gluPickMatrix(MouseX, selection_viewport[3] - MouseY, 1.0, 1.0, selection_viewport);
 gluPerspective(ViewAngle, 1.0, Front, Back);

 set_lookat();

 //識別番号を割り振りながら pick 用の描画（画面には表示されない）．球の作成のみでよい
 glRenderMode(GL_SELECT);
 glInitNames();
 glPushName(0);

 int cnt = SphereVect.size();
 for(int i = 0; i != cnt; ++i) {
  glLoadName(SphereVect[i].SerialNum);
  glPushMatrix();
  glTranslated(
        (*SphereVect[i].Pos)[0],
        (*SphereVect[i].Pos)[1],
        (*SphereVect[i].Pos)[2]);
  gluSphere(QSurface, SphereVect[i].Radius, Slices, Stacks);
  glPopMatrix();
 }

 //GLuint picked = std::numeric_limits<int>::max();
 GLuint picked = -1;
 GLint hit = glRenderMode(GL_RENDER); //ヒットしたオブジェクト数
 //ヒットしなければ UINT_MAX が返る
 if(hit) {
  //最も視点に近いオブジェクトを抽出．選択バッファの最初の要素より近いものを探す
  GLuint pos = 1;
  GLuint min = selection[pos];
  picked = selection[pos + 2];
  for(GLint i = 1; i < hit; i++) {
   pos += 4;
   if(min > selection[pos]) {
    min = selection[pos];

    picked = selection[pos + 2];
   }
  }
 }
 delete[] selection;
 set_projection();
 set_lookat();

 return picked;
}

double Builcule_Draw::measure_distance() {
 if(PickedSerial.size() != 2) {
  QMessageBox::warning(this, "エラー", "2 原子をピックしてください．");
  return -10.0;
 }

 int idx0 = SerialToSphereIdx[PickedSerial[0]];
 int idx1 = SerialToSphereIdx[PickedSerial[1]];
 const Eigen::Vector3d *xyz0 = SphereVect[idx0].Pos;
 const Eigen::Vector3d *xyz1 = SphereVect[idx1].Pos;
 return (*xyz1 - *xyz0).norm();
}

double Builcule_Draw::measure_angle() {
 if(PickedSerial.size() != 3) {
  QMessageBox::warning(this, "エラー", "3 原子をピックしてください．");
  return -10.0 * M_PI;
 }

 int idx0 = SerialToSphereIdx[PickedSerial[0]];
 int idx1 = SerialToSphereIdx[PickedSerial[1]];
 int idx2 = SerialToSphereIdx[PickedSerial[2]];
 const Eigen::Vector3d *v0 = SphereVect[idx0].Pos;
 const Eigen::Vector3d *v1 = SphereVect[idx1].Pos;
 const Eigen::Vector3d *v2 = SphereVect[idx2].Pos;
 //同名の関数なのでグローバルスコープを明示する必要がある
 return ::measure_angle(v0, v1, v2);
}

double Builcule_Draw::measure_diheadral() {
 if(PickedSerial.size() != 4) {
  QMessageBox::warning(this, "エラー", "4 原子をピックしてください．");
  return -10.0 * M_PI;
 }

 int idx0 = SerialToSphereIdx[PickedSerial[0]];
 int idx1 = SerialToSphereIdx[PickedSerial[1]];
 int idx2 = SerialToSphereIdx[PickedSerial[2]];
 int idx3 = SerialToSphereIdx[PickedSerial[3]];
 const Eigen::Vector3d *v0 = SphereVect[idx0].Pos;
 const Eigen::Vector3d *v1 = SphereVect[idx1].Pos;
 const Eigen::Vector3d *v2 = SphereVect[idx2].Pos;
 const Eigen::Vector3d *v3 = SphereVect[idx3].Pos;
 //同名の関数なのでグローバルスコープを明示する必要がある
 return ::measure_diheadral(v0, v1, v2, v3);
}

void Builcule_Draw::change_elmt(int elmt) {
 for(int serial : PickedSerial) {
  int idx = SerialToSphereIdx[serial];
  SphereVect[idx].Elmt = elmt;
  const std::string *symbol =  SharedData.get_symbol(elmt);
  SphereVect[idx].Symbol = QString(symbol->data());
 }
}

int Builcule_Draw::get_mol_idx(int serial) const {
 if(SerialToSphereIdx.find(serial) == SerialToSphereIdx.end())
  return -1;
 int idx = SerialToSphereIdx.at(serial);
 return SphereVect[idx].MolIdx;
}

int Builcule_Draw::get_pept_idx(int serial) const {
 if(SerialToSphereIdx.find(serial) == SerialToSphereIdx.end())
  return -1;
 int idx = SerialToSphereIdx.at(serial);
 return SphereVect[idx].PeptIdx;
}

int Builcule_Draw::get_seq_idx(int serial) const {
 if(SerialToSphereIdx.find(serial) == SerialToSphereIdx.end())
  return -1;
 int idx = SerialToSphereIdx.at(serial);
 return SphereVect[idx].SeqIdx;
}

