#include "selector_aa.h"
#include <QHeaderView>
#include <QLabel>

Selector_AA::Selector_AA(QWidget *parent)
 : QWidget(parent)
{
 LayoutCentral = new QVBoxLayout(this);

 //サイズ設定は Builcule_Main のコンストラクタで
 QLabel *label_aa = new QLabel(tr("アミノ酸リスト"), this);
 LayoutCentral->addWidget(label_aa);

 //リスト
 TableWidgetAA = new QTableWidget(this);
 TableWidgetAA->setColumnCount(4);
 //カラムの幅は，この文字列の幅で決まる
 TableWidgetAA->setHorizontalHeaderLabels(
     QStringList() << "mol " << "pept" << "  seq  " << "code");

 TableWidgetAA->setSelectionMode(
        QAbstractItemView::ExtendedSelection);
 TableWidgetAA->setSelectionBehavior(
        QAbstractItemView::SelectRows);
 TableWidgetAA->resizeColumnsToContents();
 TableWidgetAA->verticalHeader()->setVisible(false);

 LayoutCentral->addWidget(TableWidgetAA);

 connect(TableWidgetAA, SIGNAL(itemSelectionChanged()),
        this, SLOT(slot_selection_updated()));

 QPushButton *button_clear = new QPushButton(tr("選択を解除"), this);
 button_clear->setToolTip("選択を解除します");
 LayoutCentral->addWidget(button_clear);
 connect(button_clear, SIGNAL(clicked()),
      this, SLOT(slot_selection_clear()));

 QPushButton *button_invert = new QPushButton(tr("選択を反転"), this);
 button_invert->setToolTip("選択 / 非選択を交換します");
 LayoutCentral->addWidget(button_invert);
 connect(button_invert, SIGNAL(clicked()),
      this, SLOT(slot_selection_invert()));

 QPushButton *button_delete = new QPushButton(tr("選択を欠失"), this);
 button_delete->setToolTip("選択されたアミノ酸を削除します");
 LayoutCentral->addWidget(button_delete);
 connect(button_delete, SIGNAL(clicked()),
      this, SLOT(slot_selection_delete()));

 QPushButton *button_mutate = new QPushButton(tr("選択を変異"), this);
 button_mutate->setToolTip("選択されたアミノ酸を変異します");
 LayoutCentral->addWidget(button_mutate);
 connect(button_mutate, SIGNAL(clicked()),
      this, SLOT(slot_selection_mutate()));

 //サイズポリシー
 LayoutCentral->setContentsMargins(0, 0, 0, 0);
}

void Selector_AA::set_item(
        const std::vector<std::vector<AA_Memo>> *aa_data) {
 //アミノ酸の合計．二次元データなので総計を求める
 int pept_cnt = aa_data->size();
 int aa_cnt = 0;
 for(const auto &row : *aa_data)
  aa_cnt += row.size();

 //表示領域を確保
 TableWidgetAA->setRowCount(aa_cnt);
 TableWidgetAA->setColumnCount(4);

 //表示アイテムと SearchMap の作成
 SearchMap.clear();

 int row = 0;
 int mol(0), pept(0), seq(0);
 for(int i = 0; i <pept_cnt; ++i) {
  mol = (*aa_data)[i][0].MolNum;
  pept = (*aa_data)[i][0].PeptNum;

  aa_cnt = (*aa_data)[i].size();
  for(int j = 0; j != aa_cnt; ++j) {
   seq = (*aa_data)[i][j].SeqNum;

   //分子番号
   QTableWidgetItem *item0 = new QTableWidgetItem();
   item0->setFlags(item0->flags() ^ Qt::ItemIsEditable);
   item0->setData(Qt::DisplayRole, mol);
   TableWidgetAA->setItem(row, 0, item0);

   //ペプチド番号
   QTableWidgetItem *item1 = new QTableWidgetItem();
   item1->setFlags(item1->flags() ^ Qt::ItemIsEditable);
   item1->setData(Qt::DisplayRole, pept);
   TableWidgetAA->setItem(row, 1, item1);

   //アミノ酸番号
   QTableWidgetItem *item2 = new QTableWidgetItem();
   item2->setFlags(item2->flags() ^ Qt::ItemIsEditable);
   item2->setData(Qt::DisplayRole, seq);
   TableWidgetAA->setItem(row, 2, item2);

   //アミノ酸コード
   QTableWidgetItem *item3 = new QTableWidgetItem();
   item3->setFlags(item3->flags() ^ Qt::ItemIsEditable);
   char code1 = (*aa_data)[i][j].Code;
   const std::string *code3 = SharedData.get_aa_code_3(code1);
   //下では char* が QString にキャストされる
   item3->setData(Qt::DisplayRole, code3->data());
   TableWidgetAA->setItem(row, 3, item3);

   SearchMap[std::array<int, 3>{mol, pept, seq}] = row++;
  }
 }
}

void Selector_AA::slot_selection_clear() {
 Updating = true;

 int cnt = TableWidgetAA->rowCount();
 for(int i = 0; i != cnt; ++i)
  for(int j = 0; j < 4; ++j)
   if(TableWidgetAA->item(i, 0)->isSelected()) {
    for(int j = 0; j < 4; ++j)
     TableWidgetAA->item(i, j)->setSelected(false);
   }

 Updating = false;
 slot_selection_updated();
}

void Selector_AA::slot_selection_invert() {
 Updating = true;

 int cnt = TableWidgetAA->rowCount();
 for(int i = 0; i < cnt; ++i) {
  //第 0 列の選択/非選択を他の列に適用
  if(TableWidgetAA->item(i, 0)->isSelected()) {
   for(int j = 0; j < 4; ++j)
    TableWidgetAA->item(i, j)->setSelected(false);
  }
  else {
   for(int j = 0; j < 4; ++j)
    TableWidgetAA->item(i, j)->setSelected(true);
  }
 }

 Updating = false;
 slot_selection_updated();
}

void Selector_AA::select(
        const std::vector<std::array<int, 3>> *vect) {
 Updating = true;

 for(const auto &array : *vect)
  select(array);

 Updating = false;
 slot_selection_updated();
}

void Selector_AA::select(const std::array<int, 3> &key) {
 int idx = SearchMap[key];
 for(int i = 0; i < 4; ++i)
  TableWidgetAA->item(idx, i)->setSelected(true);
}

void Selector_AA::slot_selection_updated() {
 if(Updating)
  return;

 int cnt = TableWidgetAA->rowCount();
 for(int i = 0; i < cnt; ++i) {
  QTableWidgetItem *item = TableWidgetAA->item(i, 0);
  if(item->isSelected())
   SelectedIdx.push_back(i);
 }

 emit_selection_updated();
}

const std::vector<AA_Memo> *Selector_AA::get_selected_aa_data() {
 SelectedAAData.clear();
 SelectedAAData.shrink_to_fit();
 AA_Memo aa;

 int cnt = TableWidgetAA->rowCount();
 for(int i = 0; i < cnt; ++i) {
  QTableWidgetItem *item = TableWidgetAA->item(i, 0);
  if(item->isSelected()) {
   aa.MolNum = (item->data(Qt::DisplayRole)).toInt();
   item = TableWidgetAA->item(i, 1);
   aa.PeptNum = (item->data(Qt::DisplayRole)).toInt();
   item = TableWidgetAA->item(i, 2);
   aa.SeqNum = (item->data(Qt::DisplayRole)).toInt();
   item = TableWidgetAA->item(i, 3);
   QString qcode3 = (item->data(Qt::DisplayRole)).toString();
   const std::string tmp = qcode3.toStdString();
   aa.Code = SharedData.get_aa_code_1(&tmp);
   SelectedAAData.push_back(aa);
  }
 }
 return &SelectedAAData;
}

