#include "window_align_base.h"
#include "window_show_score.h"
#include <QDesktopWidget>
#include <QFileDialog>
#include <QGroupBox>
#include <QMessageBox>
#include <QPushButton>
#include <QSizePolicy>

Window_Align_Base::Window_Align_Base(
        QWidget *parent,
        Builcule_Unit *unit1,
        Config_File *config,
        Window_Sup *sup)
 : QMainWindow(parent),
   AlignNum(0),
   Unit1(unit1),
   ConfigFile(config),
   PeptIdx0(-1),
   PeptIdx1(-1),
   Mol0(-1),
   Pept0(-1),
   Mol1(-1),
   Pept1(-1),
   AaRepl(0),
   GapIni(8),
   GapElg(2),
   Window(1),
   MenuBar(menuBar()),
   WinSup(sup)
{
 //ウィンドウを表示しているあいだは，メインウィンドウでの操作を禁止する
 //ウィジェットクラスでは動作しない
 setWindowModality(Qt::WindowModal);
 setWindowFlags(windowFlags() | Qt::WindowStaysOnTopHint);

 //ファイルメニュー
 MenuFile = new QMenu("ファイル(&F)", this);
 MenuBar->addMenu(MenuFile);

 QAction *action_save_png
        = new QAction("PNG 形式の画像として保存", MenuBar);
 connect(action_save_png, SIGNAL(triggered()),
        this, SLOT(slot_save_png()));
 MenuFile->addAction(action_save_png);

 ActionHide = new QAction("閉じる(&C)", MenuBar);
 connect(ActionHide, SIGNAL(triggered()), this, SLOT(hide()));
 MenuFile->addAction(ActionHide);

 //計算メニュー
 MenuCalc = new QMenu("計算(&C)", this);
 MenuBar->addMenu(MenuCalc);

 QAction *action_alignment
        = new QAction("アラインメント", MenuBar);
 connect(action_alignment, SIGNAL(triggered()),
        this, SLOT(slot_alignment()));
 MenuCalc->addAction(action_alignment);

 //表示メニュー
 MenuShow = new QMenu("表示(&V)", this);
 MenuBar->addMenu(MenuShow);

 QAction *action_show_score
        = new QAction("スコア", MenuBar);
 connect(action_show_score, SIGNAL(triggered()),
        this, SLOT(slot_show_score_mtx()));
 MenuShow->addAction(action_show_score);

 //ウィンドウサイズ．これに載せる Widget_Show_Align との兼ね合い
 //スクリーンの解像度より大きい場合は最大化
 QDesktopWidget screen;
 int s_width = screen.width();
 int s_height = screen.height();
 int d_width = 1400;
 int d_height = 700;

 if(s_width < d_width || s_height < d_height)
  setWindowState(windowState() ^ Qt::WindowMaximized);
 else
  setMinimumSize(d_width, d_height);

 //セントラルウィジェット
 QWidget *widget_central = new QWidget(this);
 widget_central->setMaximumSize(250, 550);
 widget_central->setSizePolicy(
    QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed));
 setCentralWidget(widget_central);
 LayoutCentral = new QVBoxLayout(widget_central);

 //ペプチド選択用コンボボックスを 2 個横縦びに（セントラルウィジェットに載せる）
 QGroupBox *gbox_pept
        = new QGroupBox("ペプチドの選択", widget_central);
 QVBoxLayout *layout_combo = new QVBoxLayout(gbox_pept);

 //LabelC0 の文字列は継承クラスで設定
 layout_combo->addWidget(&LabelC0);
 layout_combo->addWidget(&PeptCombo[0]);

 QLabel *label_c1 = new QLabel(this);
 label_c1->setText("ユニット 1");
 layout_combo->addWidget(label_c1);
 layout_combo->addWidget(&PeptCombo[1]);

 //テンプレートペプチドは各クラスで共通なので，ここで作成できる
 create_pept_combo(1, Unit1);

 //パラメータの初期値はここで設定

 //アミノ酸置換行列の選択
 QGroupBox *gbox_repl
        = new QGroupBox("アミノ酸置換行列の選択", widget_central);
 QHBoxLayout *layout_repl = new QHBoxLayout(gbox_repl);

 RadioButtonRepl[0].setText("PAM250");
 layout_repl->addWidget(&RadioButtonRepl[0]);
 GroupAaRepl.addButton(&RadioButtonRepl[0], 0);

 RadioButtonRepl[1].setText("BLOSUM62");
 layout_repl->addWidget(&RadioButtonRepl[1]);
 GroupAaRepl.addButton(&RadioButtonRepl[1], 1);

 RadioButtonRepl[AaRepl].setChecked(true);

 //ギャップの設定
 QGroupBox *gbox_gap
        = new QGroupBox("ギャップの設定", widget_central);
 QGridLayout *layout_gap = new QGridLayout(gbox_gap);

 QLabel *lb_ini = new QLabel(widget_central);
 lb_ini->setText("ギャップ開始ペナルティ");

 QLabel *lb_elong = new QLabel(widget_central);
 lb_elong->setText("ギャップ伸長ペナルティ");

 layout_gap->addWidget(lb_ini, 0, 0);
 layout_gap->addWidget(&LeIni, 0, 1);
 layout_gap->addWidget(lb_elong, 1, 0);
 layout_gap->addWidget(&LeElg, 1, 1);

 LeIni.insert(QString::number(GapIni));
 LeElg.insert(QString::number(GapElg));

 //左ウィジェットのレイアウト
 LayoutCentral->addWidget(gbox_pept);
 LayoutCentral->addWidget(gbox_repl);
 LayoutCentral->addWidget(gbox_gap);

 //空のドックウィジェット
 WidgetAlign = new Widget_Show_Align(this, AlignNum);
 DockRight = new QDockWidget();
 DockRight->setWidget(WidgetAlign);
 addDockWidget(Qt::RightDockWidgetArea, DockRight);

 widget_central->setSizePolicy(
    QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed));
}

void Window_Align_Base::create_pept_combo(
        int unit_num, Builcule_Unit *unit) {
 PeptNum[unit_num].clear();
 PeptNum[unit_num].shrink_to_fit();

 //ペプチド情報を作成．コンボボックスのラベルとペプチドの指定に使う
 std::array<int, 2> row {-1, -1};  //分子番号とペプチド番号
 int mol_cnt = unit->get_mol_cnt();
 for(int i = 0; i != mol_cnt; ++i) {
  row[0] = i;
  int pept_cnt = unit->get_pept_cnt(i);
  for(int j = 0; j < pept_cnt; ++j) {
   row[1] = j;
   PeptNum[unit_num].push_back(row);
  }
 }

 //コンボボックスのラベルを作成
 for(const std::array<int, 2> &row : PeptNum[unit_num])
  PeptCombo[unit_num].addItem(
        QString("分子番号：")
        + QString::number(row[0])
        + QString(" ペプチド番号：")
        + QString::number(row[1]));
}

bool Window_Align_Base::capture_prm_base() {
 PeptIdx1 = PeptCombo[1].currentIndex();
 Mol1 = PeptNum[1][PeptIdx1][0];
 Pept1 = PeptNum[1][PeptIdx1][1];

 AaRepl = GroupAaRepl.checkedId();

 QString str = LeIni.text();
 GapIni = str.toDouble();
 if(GapIni <= 0.0) {  //空欄だと 0.0 になる
  QMessageBox::critical(this, "エラー",
        "ギャップ開始ペナルティが異常値です．");
  return false;
 }

 str = LeElg.text();
 GapElg = str.toDouble();
 if(GapElg <= 0.0) {
  QMessageBox::critical(this, "エラー",
        "ギャップ伸長ペナルティが異常値です．");
  return false;
 }

 return true;
}

void Window_Align_Base::set_data_base() {
 Align.clear();
 Align.set_seq_xyz(1, Unit1->get_pept(Mol1, Pept1));
 Align.set_aa_repl(AaRepl);
 Align.set_gap(GapIni, GapElg);
}

void Window_Align_Base::slot_show_score_mtx() {
 const std::vector<std::vector<double>> *mtx
        = Align.get_score_mtx();
 if(mtx->empty()) {
  QMessageBox::information(this, "情報", "まずアラインメントを計算してください．");
  return;
 }

 Window_Show_Score *win_score = new Window_Show_Score(this);
 win_score->show();
 win_score->set_score_mtx(mtx);
 win_score->repaint();
}

void Window_Align_Base::trace_back() {
 if(Align.get_score_mtx()->empty()) {
  QMessageBox::information(this, "情報", "スコアが計算できません．");
  return;
 }

 Align.trace_back();
 show_align();
}

void Window_Align_Base::show_align() {
 const auto *alignment = Align.get_alignment();
 if(alignment->empty()) {
  QMessageBox::information(this, "情報", "アラインメントが作成されていません．");
  return;
 }

 WidgetAlign = new Widget_Show_Align(this, AlignNum);
 WidgetAlign->set_alignment(alignment);
 WidgetAlign->set_score(Align.get_score());
 WidgetAlign->set_ini_ter(Align.get_ini_ter());
 WidgetAlign->show_align();

 QDesktopWidget screen;
 int s_width = screen.width();
 int s_height = screen.height();
 int d_width = 1400;
 int d_height = 700;
 if(s_width > d_width && s_height > d_height)
  WidgetAlign->set_scroll_area_size(1100, 600);

 //Dock に入れると WidgetAlign-> でアクセスできなくなる
 removeDockWidget(DockRight);
 DockRight = new QDockWidget();
 DockRight->setWidget(WidgetAlign);
 addDockWidget(Qt::RightDockWidgetArea, DockRight);
}

void Window_Align_Base::slot_save_png() {
 QFileDialog dialog(this);

 dialog.setAcceptMode(QFileDialog::AcceptSave);
 dialog.selectFile("*.png");
 dialog.setNameFilter("png 形式 *.png");
 if(!dialog.exec())  //enum QDialog::DialogCode が返る
  return;

 QStringList path_list = dialog.selectedFiles();
 QString qpath = path_list[0];
 QFile file(qpath);
 const std::string path = qpath.toStdString();

 if(!WidgetAlign->grab().save(qpath)) {
  QMessageBox::critical(this, "エラー", "保存中にエラーが発生しました．");
 }
}

