#include "detrial_draw.h"
#include <QString>
#include <QVBoxLayout>

Detrial_Draw::Detrial_Draw(QWidget *parent)
 : QOpenGLWidget(parent),
   unitNum(-1),
   drawCenter(Eigen::Vector3d::Zero(3)),
   drawRadius(0.0),
   eyePos(Eigen::Vector3d::Zero(3)),
   Front(0.0),
   Back(0.0),
   Slices(0),
   Stacks(0),
   QSurface(gluNewQuadric()),  //二次曲面オブジェクトを生成し．そのポインタを設定
   mouseX(0),
   mouseY(0),
   mouseMoveLength(0.0),
   pressedButton(Qt::NoButton),
   sharedPrm(nullptr),
   atomSphere(nullptr),
   bondCylinder(nullptr),
   peptTriangle(nullptr),
   atomVect(nullptr),
   molVect(nullptr),
   altDraw(nullptr),
   altAtomVect(nullptr)
{
 popupWidget = new QWidget(this);
 popupWidget->setWindowFlags(Qt::ToolTip);
 QVBoxLayout *popup_layout = new QVBoxLayout(popupWidget);
 popupLabel = new QLabel(popupWidget);
 popupLabel->setStyleSheet("background-color:white;");
 popup_layout->addWidget(popupLabel);
 popupWidget->setLayout(popup_layout);
}

void Detrial_Draw::clear() {
 drawCenter << 0.0, 0.0, 0.0;
 drawRadius= 0.0;
 eyePos << 0.0, 0.0, 0.0;
 Front = 0.0;
 Back = 0.0;
 Slices = 0;
 Stacks = 0;
 mouseX = 0;
 mouseY = 0;
 mouseMoveLength = 0.0;
 pressedButton = Qt::NoButton;
}

void Detrial_Draw::set_alt(std::vector<Atom> *vect, Detrial_Draw *draw) {
 altAtomVect = vect;
 altDraw = draw;
}

void Detrial_Draw::init_eye_pos() {
 eyePos[0] = drawCenter[0];
 eyePos[1] = drawCenter[1];
 eyePos[2] = drawCenter[2] + drawRadius * 10.0;

 Front = drawRadius * 9.0;
 Back = drawRadius * 11.0;
}

void Detrial_Draw::initializeGL() {
 glClearColor(1.0, 1.0, 1.0, 1.0); //背景を白色に設定
 glEnable(GL_DEPTH_TEST);
}

void Detrial_Draw::resizeGL(int w, int h) {
 GLsizei length = (w > h) ? h : w;
 resize(length, length);  //描画領域を正方形に保つ
}

void Detrial_Draw::set_projection() {
 //射影変換
 glMatrixMode(GL_PROJECTION);
 glLoadIdentity();
 //field of view, aspect ratio, depth range
 gluPerspective(viewAngle, 1.0, Front, Back);
}

void Detrial_Draw::set_lookat() {
 //モデルビュー変換
 glMatrixMode(GL_MODELVIEW);
 glLoadIdentity();
 //view point, focus point, up vector
 gluLookAt(eyePos[0], eyePos[1], eyePos[2],
        drawCenter[0], drawCenter[1], drawCenter[2],
        0.0, 1.0, 0.0);
}

void Detrial_Draw::paintGL() {  //override 関数
 if(!atomVect || atomVect->empty()) {
  return;
 }

 glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

 if(Front < 10.0)
  Slices = 64;
 else if(Front < 100.0)
  Slices = 32;
 else
  Slices = 16;

 Stacks = Slices / 2;
/*
 int slices0 = 0;
 if(Front != 0)
  slices0 = 6000 / abs(Front);
 if(Front == 0 || slices0 > 24)  //細かく分割すると動作が重くなる
  slices0 = 24;

 Stacks = Slices = slices0;
*/
 set_projection();
 set_lookat();

 glEnable(GL_LIGHTING);
 glEnable(GL_LIGHT0);
 glEnable(GL_CULL_FACE);
 glCullFace(GL_BACK);

 gluQuadricDrawStyle(QSurface, GLU_FILL);
 if(sharedPrm->pickMode)
  draw_sphere(2);

 GLenum style = sharedPrm->Quadric[unitNum][0];
 if(style == GLU_POINT)
  Stacks = Slices = Slices * 2 / 3;
 else if(style == GLU_LINE)
  Stacks = Slices = Slices / 2;

 gluQuadricDrawStyle(QSurface, style);
 MODEL model = DEFAULT_MODEL;
 int model_cnt = sharedPrm->Model[unitNum][0].size();
 for(int i = 0; i < model_cnt; ++i) {
  model = sharedPrm->Model[unitNum][0][i];
  if(model == VDW || model == BALL || model == ROD)
   draw_sphere(0);
  if(model == STICK || model == ROD)
   draw_cylinder(0);
  if(model == WIRE)
   draw_wire(0);
  if(model == RIBBON)
   draw_triangle();
 }

 if(sharedPrm->drawDivideMode) {
  GLenum style = sharedPrm->Quadric[unitNum][1];
  if(style == GLU_POINT)
   Stacks = Slices = Slices * 2 / 3;
  else if(style == GLU_LINE)
   Stacks = Slices = Slices / 2;

  gluQuadricDrawStyle(QSurface, style);
  model_cnt = sharedPrm->Model[unitNum][1].size();
  for(int i = 0; i < model_cnt; ++i) {
   model = sharedPrm->Model[unitNum][1][i];
   if(model == VDW || model == BALL || model == ROD)
    draw_sphere(1);
   if(model == STICK || model == ROD)
    draw_cylinder(1);
   if(model == WIRE)
    draw_wire(1);
   if(model == RIBBON)
    draw_triangle();
  }
 }
}

void Detrial_Draw::draw_sphere(int idx) {
 const Eigen::Vector3d *xyz = nullptr;
 for(const Sphere &sphere : (*atomSphere)[idx]) {
  if(sphere.Color[3] > 0.0f) {
   xyz = sphere.Pos;
   glMaterialfv(GL_FRONT, GL_DIFFUSE, sphere.Color);
   glPushMatrix();
   glTranslated((*xyz)[0], (*xyz)[1], (*xyz)[2]);
   gluSphere(QSurface, sphere.Radius, Slices, Stacks);
   glPopMatrix();
  }
 }
}

void Detrial_Draw::draw_cylinder(int idx) {
 Eigen::Vector3d tmp_vect{0.0, 0.0, 0.0};
 Eigen::Vector3d tmp_axis{0.0, 0.0, 0.0};
 for(const Cylinder &cylinder : (*bondCylinder)[idx]) {
  if(cylinder.Color[3] > 0.0f) {
   tmp_vect = (*cylinder.Pos1) - (*cylinder.Pos0);
   double norm = tmp_vect.norm();
   tmp_vect = tmp_vect.normalized();
   //結合と Z 軸との成す角を内積 (z 軸 = 0.0, 0.0, 1.0) から求める
   double angle = acos(tmp_vect[2]) * RAD_to_DEG;
   //回転軸を外積から求める
   tmp_axis = zAxis.cross(tmp_vect).normalized();
   const Eigen::Vector3d &pos = (*cylinder.Pos0);
   glMaterialfv(GL_FRONT, GL_DIFFUSE, cylinder.Color);
   glPushMatrix();
   glTranslated(pos[0], pos[1], pos[2]);  //シリンダを平行移動
   glRotated(angle, tmp_axis[0], tmp_axis[1], tmp_axis[2]);  //同回転
   gluCylinder(QSurface, cylinder.Radius, cylinder.Radius, norm, Slices, 1);
   glPopMatrix ();
  }
 }
}

void Detrial_Draw::draw_triangle() {
 glDisable(GL_LIGHTING);
 glDisable(GL_CULL_FACE);

 glBegin(GL_TRIANGLES);
 for(const Triangle tri: *peptTriangle) {
  glColor4f(tri.Color[0], tri.Color[1], tri.Color[2], tri.Color[3]);
  glVertex3f((*tri.Pos0)[0], (*tri.Pos0)[1], (*tri.Pos0)[2]);
  glColor4f(tri.Color[0], tri.Color[1], tri.Color[2], tri.Color[3]);
  glVertex3f((*tri.Pos1)[0], (*tri.Pos1)[1], (*tri.Pos1)[2]);
  glColor4f(tri.Color[0], tri.Color[1], tri.Color[2], tri.Color[3]);
  glVertex3f((*tri.Pos2)[0], (*tri.Pos2)[1], (*tri.Pos2)[2]);
 }
  glEnd();

 glEnable(GL_LIGHTING);
 glEnable(GL_LIGHT0);
 glEnable(GL_CULL_FACE);
 glCullFace(GL_BACK);
}

void Detrial_Draw::draw_wire(int idx) {
 glDisable(GL_LIGHTING);
 glDisable(GL_CULL_FACE);

 glBegin(GL_LINES);
 for(const Cylinder line: (*bondCylinder)[idx]) {
  glColor4f(line.Color[0], line.Color[1], line.Color[2], line.Color[3]);
  glVertex3f((*line.Pos0)[0], (*line.Pos0)[1], (*line.Pos0)[2]);
  glVertex3f((*line.Pos1)[0], (*line.Pos1)[1], (*line.Pos1)[2]);
 }
 glEnd();

 glEnable(GL_LIGHTING);
 glEnable(GL_LIGHT0);
 glEnable(GL_CULL_FACE);
 glCullFace(GL_BACK);
}

void Detrial_Draw::mousePressEvent(QMouseEvent *event) {
 if(!atomVect || atomVect->empty()) return;
 if(sharedPrm->KeyZ) return;

 //回転用の設定
 GLsizei length = (width() > height()) ? height() : width();
 mouseMoveLength = 2.0 * drawRadius / (double)length;

 pressedButton = event->button();
 mouseX = event->x();
 mouseY = event->y();

 //ピック用の設定
 if(!sharedPrm->pickMode)
  return;

 if(event->button() != Qt::LeftButton)
  return;

 int picked_serial = pick();
 if(picked_serial == std::numeric_limits<int>::max())
  return;

 //ピックされた原子の情報を表示
 int atom_idx = serialToIdx->at(picked_serial);
 const Atom &atom = (*atomVect)[atom_idx];
 QString info(QString::fromStdString(atom.Symbol));
 info += QString("\n分子番号：") += QString::number(atom.molIdx) ;
 if(atom.peptIdx > -1) {
  info += QString("\nペプチド番号：") += QString::number(atom.peptIdx);
  info += QString("\n配列番号：") += QString::number(atom.aaIdx);
  info += QString("\nアミノ酸コード：") += atom.aaCode;
 }
 popupLabel->setText(info);
 popupWidget->move(event->globalX() + 10, event->globalY() + 10);
 popupWidget->show();

 int pos = picked_pos(picked_serial);
 if(pos == -1) {  //未ピックである原子をピックした場合
  pick(serialToIdx->at(picked_serial));
 }
 else {  //ピック済である原子をピックした場合
  (*atomSphere)[2].erase((*atomSphere)[2].begin() + pos);
  if(sharedPrm->pickMode)
   repaint();
 }
}

int Detrial_Draw::picked_pos(int serial) {
 int cnt = (*atomSphere)[2].size();
 for(int i = 0; i < cnt; ++i) {
  if((*atomSphere)[2][i].Serial == serial) {
   return i;
  }
 }
 return -1;
}

void Detrial_Draw::pick(int idx) {
 //atomVect から作成して push
 Sphere sphere;
 sphere.Serial = (*atomVect)[idx].serialNum;
 sphere.Pos = &(*atomVect)[idx].XYZ;
 sphere.Elmt = (*atomVect)[idx].Elmt;
 sphere.Radius = ATOM_DISTANCE.vdw_radius()[sphere.Elmt];
 for(int i = 0; i != 3; ++i)
  sphere.Color[i] = atomColor[120][i];
 (*atomSphere)[2].push_back(sphere);
 if(sharedPrm->pickMode)
  repaint();
}

void Detrial_Draw::mouseMoveEvent(QMouseEvent *event) {
 if(!atomVect || atomVect->empty()) return;
 if(sharedPrm->KeyZ) return;

 if(sharedPrm->KeyA) {
  if(altAtomVect && altDraw) {  //両ユニットを回転
   rotate(0, event->x(), event->y());
   rotate(1, event->x(), event->y());
   altDraw->repaint();
  }
 }
 else if(sharedPrm->KeyX) {  //視点を移動
  move_eye(event->x(), event->y());
 }
 else if(pressedButton == Qt::LeftButton) {
  rotate(unitNum, event->x(), event->y());
 }

 repaint();
 mouseX = event->x();
 mouseY = event->y();
}

void Detrial_Draw::mouseReleaseEvent(QMouseEvent *event) {
 if(!atomVect || atomVect->empty()) return;

 mouseX = event->x();
 mouseY = event->y();

 unitAffine[unitNum].init_center();

 if(popupWidget->isVisible())
  popupWidget->hide();
}

void Detrial_Draw::move_eye(int ox, int oy) {
 double dx = (double)(mouseX - ox) / (double)width() * drawRadius;
 double dy = (double)(mouseY - oy) / (double)height() * drawRadius;
 drawCenter[0] += dx;
 drawCenter[1] -= dy;
 eyePos[0] += dx;
 eyePos[1] -= dy;
}

void Detrial_Draw::rotate(int unit, int ox, int oy) {
 //縦方向の動作で X 軸を，横方向の動作で Y 軸を回転
 double dx = (double)(ox - mouseX) / (double)height() * 360.0;
 double dy = (double)(oy - mouseY) / (double)width() * 360.0;
 unitAffine[unit].rotate_unit(dx, dy);
}

void Detrial_Draw::wheelEvent(QWheelEvent *event) {
 if(!atomVect || atomVect->empty())
  return;
 if(!sharedPrm->KeyZ)
  return;

 double d = event->angleDelta().y();
 d *= drawRadius;
 d /= 500.0;

 if(eyePos[2] >= 0.0)
  eyePos[2] -= d;
 else
  eyePos[2] += d;
 Front -= d;
 Back -= d;
 repaint();
}

int Detrial_Draw::pick() {
 GLuint picked = std::numeric_limits<int>::max();
 if(!atomVect || atomVect->empty())
  return picked;

 //ビューポートを設定する箇所はここだけでよい
 glViewport(0, 0, width(), height());

 //クリック位置に描画されている原子を格納するための配列を確保
 const GLuint selBufferSize(400);  //1 原子につき配列 4 個分を利用する．ここでは 100 原子分
 GLuint *selection = new GLuint[selBufferSize];
 glSelectBuffer(selBufferSize, selection);
 GLint selection_viewport[4];
 glGetIntegerv(GL_VIEWPORT, selection_viewport);

 //ここは，gluPickMatrix() の呼び出しがあるので，set_projection(); に書き換えられない．保守に注意
 glMatrixMode(GL_PROJECTION);
 glLoadIdentity();
 gluPickMatrix(mouseX, selection_viewport[3] - mouseY, 1.0, 1.0, selection_viewport);
 gluPerspective(viewAngle, 1.0, Front, Back);

 set_lookat();

 //識別番号を割り振りながら pick 用の描画（画面には表示されない）．球の作成のみでよい
 glRenderMode(GL_SELECT);
 glInitNames();
 glPushName(0);

 MODEL model = DEFAULT_MODEL;
 int model_cnt = 0;
 if(sharedPrm->Quadric[unitNum][0] == GLU_FILL) {
  model_cnt = sharedPrm->Model[unitNum][0].size();
  for(int i = 0; i < model_cnt; ++i) {
   model = sharedPrm->Model[unitNum][0][i];
   if(model == VDW || model == BALL || model == ROD)
    draw_sphere_for_pick(0);
  }
 }

 if(sharedPrm->drawDivideMode) {
  if(sharedPrm->Quadric[unitNum][1] == GLU_FILL) {
   model_cnt = sharedPrm->Model[unitNum][1].size();
   for(int i = 0; i < model_cnt; ++i) {
    model = sharedPrm->Model[unitNum][1][i];
    if(model == VDW || model == BALL || model == ROD)
     draw_sphere_for_pick(1);
   }
 }

 if(sharedPrm->pickMode)
  draw_sphere_for_pick(2);
 }

 GLint hit = glRenderMode(GL_RENDER); //ヒットしたオブジェクト数
 //ヒットしなければ UINT_MAX が返る
 if(hit) {
  //最も視点に近いオブジェクトを抽出．選択バッファの最初の要素より近いものを探す
  GLuint pos = 1;
  GLuint min = selection[pos];
  picked = selection[pos + 2];
  for(GLint i = 1; i < hit; i++) {
   pos += 4;
   if(min > selection[pos]) {
    min = selection[pos];
    picked = selection[pos + 2];
   }
  }
 }
 delete[] selection;
 set_projection();
 set_lookat();
 return picked;
}

void Detrial_Draw::draw_sphere_for_pick(int idx) {
 int cnt = (*atomSphere)[idx].size();
 for(int i = 0; i < cnt; ++i) {
  const Sphere &sphere = (*atomSphere)[idx][i];
  if(sphere.Color[3] > 0.0) {
   const Eigen::Vector3d &xyz = *sphere.Pos;
   glLoadName(sphere.Serial);
   glPushMatrix();
   glTranslated(xyz[0], xyz[1], xyz[2]);
   gluSphere(QSurface, sphere.Radius, Slices, Stacks);
   glPopMatrix();
  }
 }
}

