#include "detrial_unit.h"

Detrial_Unit::Detrial_Unit()
 : chargeCalced(false),
   Radius_Cylinder_Rod(0.4f)
{}

void Detrial_Unit::clear() {
 Unit::clear();

 for(int i = 0; i < 3; ++i) {
  atomSphere[i].clear();
  atomSphere[i].shrink_to_fit();
 }

 for(int i = 0; i < 2; ++i) {
  bondCylinder[i].clear();
  bondCylinder[i].shrink_to_fit();
 }

 peptTriangle.clear();
 peptTriangle.shrink_to_fit();

 xyzAlign3d.clear();
 xyzAlign3d.shrink_to_fit();
}

double Detrial_Unit::measure_diheadral(int serial0, int serial1, int serial2, int serial3) {
 const Eigen::Vector3d *pos0 = &atomVect[serialToIdx[serial0]].XYZ;
 const Eigen::Vector3d *pos1 = &atomVect[serialToIdx[serial1]].XYZ;
 const Eigen::Vector3d *pos2 = &atomVect[serialToIdx[serial2]].XYZ;
 const Eigen::Vector3d *pos3 = &atomVect[serialToIdx[serial3]].XYZ;

 return ::measure_diheadral(pos0, pos1, pos2, pos3);
}

void Detrial_Unit::create_xyz_for_align3d(int mol, int pept) {
 xyzAlign3d.clear();
 xyzAlign3d.shrink_to_fit();

 Eigen::Vector3d v = Eigen::Vector3d::Zero(3);
 std::vector<Eigen::Vector3d> row(4, v);
 int row_cnt = molVect[mol].aaTable[pept].size();
 xyzAlign3d.assign(row_cnt, row);

 std::vector<Amino_Acid> &pept_vect = molVect[mol].aaTable[pept];
 for(int i = 0; i != row_cnt; ++i) {
  const Amino_Acid &aa = pept_vect[i];

  int serial = aa.Annot.at(N);
  int idx = serialToIdx.at(serial);
  xyzAlign3d[i][0] = atomVect[idx].XYZ;

  serial = aa.Annot.at(CA);
  idx = serialToIdx.at(serial);
  xyzAlign3d[i][1] = atomVect[idx].XYZ;

  serial = aa.Annot.at(C);
  idx = serialToIdx.at(serial);
  xyzAlign3d[i][2] = atomVect[idx].XYZ;

  serial = aa.Annot.at(O);
  idx = serialToIdx.at(serial);
  xyzAlign3d[i][3] = atomVect[idx].XYZ;
 }
}

void Detrial_Unit::create_atom_sphere() {
 //ファイル読み込み時に VDW + CPK として作成する．[0] だけでよい
 int cnt = atomVect.size();
 atomSphere[0].assign(cnt, Sphere());
 for(int i = 0; i < cnt; ++i) {
  atomSphere[0][i].Serial = atomVect[i].serialNum;
  atomSphere[0][i].Pos = &atomVect[i].XYZ;
  atomSphere[0][i].Elmt = atomVect[i].Elmt;
  atomSphere[0][i].Radius = vdwRadius[atomVect[i].Elmt];
  for(int j = 0; j != 3; ++j)
   atomSphere[0][i].Color[j] = atomColor[atomSphere[0][i].Elmt][j];
 }
}

void Detrial_Unit::set_atom_sphere_radius(int idx, MODEL model) {
 if(model == VDW) {
  for(Sphere &sphere : atomSphere[idx])
   sphere.Radius = vdwRadius[sphere.Elmt];
 }
 else if(model == BALL) {
  for(Sphere &sphere : atomSphere[idx]) {
   if(sphere.Elmt > 1)
    sphere.Radius = ballRadiusX;
   else
    sphere.Radius = ballRadiusH;
  }
 }
 else if(model == ROD) {
  for(Sphere &sphere : atomSphere[idx])
   sphere.Radius = Radius_Cylinder_Rod;
 }
}

void Detrial_Unit::set_atom_sphere_color(int idx, COLOR color) {
 if(color == CPK) {
  for(Sphere &sphere : atomSphere[idx]) {
   for(int i = 0; i != 3; ++i)
    sphere.Color[i] = atomColor[sphere.Elmt][i];
  }
  return;
 }
 else if(color == MOL) {
  int atom_idx = -1;
  int color = 0;
  for(const Mol &mol : molVect) {
   for(int serial : mol.atomSerial) {
    color = color % 6;  //6 色用意してある
    atom_idx = serialToIdx[serial];
    for(int i = 0; i != 3; ++i)
     atomSphere[idx][atom_idx].Color[i] = molColor[color][i];
   }
   ++color;
  }
  return;
 }
 else if(color == CHARGE) {
  int atom_cnt = atomVect.size();
  for(int i = 0; i < atom_cnt; ++i) {
   if(atomVect[i].Charge < -1.0) {
    for(int j = 0; j != 4; ++j)
     atomSphere[0][i].Color[j] = molColor[0][j];
   }
   else if(atomVect[i].Charge < 0.0) {
    atomSphere[0][i].Color[0] = 1.0;
    atomSphere[0][i].Color[1] = 1.0 + atomVect[i].Charge;
    atomSphere[0][i].Color[2] = 1.0 + atomVect[i].Charge;
    atomSphere[0][i].Color[3] = 1.0;
   }
   else if(atomVect[i].Charge < 1.0) {
    atomSphere[0][i].Color[0] = 1.0 - atomVect[i].Charge;
    atomSphere[0][i].Color[1] = 1.0 - atomVect[i].Charge;
    atomSphere[0][i].Color[2] = 1.0;
    atomSphere[0][i].Color[3] = 1.0;
   }
   else {
    for(int j = 0; j != 4; ++j)
     atomSphere[0][i].Color[j] = molColor[1][j];
   }
  }
  return;
 }

 //以下は，着色しない元素もあるので，デフォルト色で初期化
 for(Sphere &sphere : atomSphere[idx])
  for(int i = 0; i != 3; ++i)
   sphere.Color[i] = atomColor[0][i];

 if(color == PEPT) {
  //ペプチドごとに色を変えて，各アミノ酸を着色
  int color_num = 0;
  int cnt = atomVect.size();
  for(int i = 0; i < cnt; ++i) {
   color_num = (atomVect[i].molIdx * 10 + atomVect[i].peptIdx) % 6;
   for(int j = 0; j != 3; ++j)
    atomSphere[idx][i].Color[j] = molColor[color_num][j];
  }
 }
 else if(color == SECONDARY) {
 //二次構造情報は Amino_Acid 内の記述を使う
  for(const Mol &mol : molVect) {
   const std::vector<std::vector<Amino_Acid>> &aa_table = mol.aaTable;
   for(const std::vector<Amino_Acid> &peptide : aa_table)
    for(const Amino_Acid &aa : peptide) {
     if(aa.Secondary == 0)
      set_atom_sphere_color_aa(idx, &aa, 0);
     else if(aa.Secondary == 2)
      set_atom_sphere_color_aa(idx, &aa, 2);
    }
  }
 }
 else if(color == AA) {
 //アミノ酸ごとに色を変えて着色
  for(const Mol &mol : molVect) {
   const std::vector<std::vector<Amino_Acid>> &aa_table = mol.aaTable;
   for(const std::vector<Amino_Acid> &peptide : aa_table) {
    for(const Amino_Acid &aa : peptide) {
     if(aa.Code1 == 'D' || aa.Code1 == 'E')
      set_atom_sphere_color_aa(idx, &aa, 0);
     else if(aa.Code1 == 'H' || aa.Code1 == 'K' || aa.Code1 == 'R')
      set_atom_sphere_color_aa(idx, &aa, 1);
     else if(aa.Code1 == 'C' || aa.Code1 == 'c')
      set_atom_sphere_color_aa(idx, &aa, 2);
     else if(aa.Code1 == 'S' || aa.Code1 == 'T' || aa.Code1 == 'N' || aa.Code1 == 'Q' || aa.Code1 == 'Y' || aa.Code1 == 'G')
      set_atom_sphere_color_aa(idx, &aa, 3);
    }
   }
  }
 }
}

void Detrial_Unit::set_atom_sphere_color_aa(int idx, const Amino_Acid *aa, int color) {
//アミノ酸の性質ごとの色分けには molColor を転用する
 for(int serial : aa->allSerial) {
  int atom_idx = serialToIdx[serial];
  for(int i = 0; i != 3; ++i)
   atomSphere[idx][atom_idx].Color[i] = molColor[color][i];
 }
}

void Detrial_Unit::create_bond_cylinder() {
 if(!bondCylinder[0].empty()) {  //作成済みなら何もしない
  return;
 }

 Cylinder cylinder;
 int atom_cnt = atomVect.size();
 for(int i = 0; i < atom_cnt; ++i) {
  cylinder.Serial0 = atomVect[i].serialNum;
  cylinder.Pos0 = &atomVect[i].XYZ;
  cylinder.molIdx = atomVect[i].molIdx;
  cylinder.peptIdx = atomVect[i].peptIdx;
  cylinder.aaCode = atomVect[i].aaCode;
  for(int pair : atomVect[i].BondUp) {
   cylinder.Serial1 = pair;
   const Atom &atom1 = atomVect[serialToIdx[pair]];
   cylinder.Pos1 = &atom1.XYZ;
   bondCylinder[0].push_back(cylinder);
  }
 }
}

void Detrial_Unit::set_bond_cylinder_radius(int idx, MODEL model) {
 if(model == STICK) {
  for(Cylinder &cylinder : bondCylinder[idx])
   cylinder.Radius = stickRadius;
 }
 else if(model == ROD) {
  for(Cylinder &cylinder : bondCylinder[idx])
   cylinder.Radius = Radius_Cylinder_Rod;
 }
}

void Detrial_Unit::set_bond_cylinder_color(int idx, COLOR color) {
//α 値は変えない
 if(color == MOL) {
  for(Cylinder &cylinder : bondCylinder[idx]) {
   int num = cylinder.molIdx % 6;
   for(int i = 0; i < 3; ++i)
    cylinder.Color[i] = molColor[num][i];
  }
 }
 else if(color == PEPT) {
  for(Cylinder &cylinder : bondCylinder[idx]) {
   int num = (cylinder.molIdx * 10 + cylinder.peptIdx) % 6;
   for(int i = 0; i < 3; ++i)
    cylinder.Color[i] = molColor[num][i];
  }
 }
 else if(color == AA) {
  for(Cylinder &cylinder : bondCylinder[idx]) {
   if(cylinder.aaCode == 'D' || cylinder.aaCode == 'E')
    for(int i = 0; i < 3; ++i)
     cylinder.Color[i] = molColor[0][i];
   else if(cylinder.aaCode == 'H' || cylinder.aaCode == 'K' || cylinder.aaCode == 'R')
    for(int i = 0; i < 3; ++i)
     cylinder.Color[i] = molColor[1][i];
   else if(cylinder.aaCode == 'C' || cylinder.aaCode == 'c')
    for(int i = 0; i < 3; ++i)
     cylinder.Color[i] = molColor[2][i];
   else if(cylinder.aaCode == 'S' || cylinder.aaCode == 'T' || cylinder.aaCode == 'N' || cylinder.aaCode == 'Q' || cylinder.aaCode == 'Y' || cylinder.aaCode == 'G')
    for(int i = 0; i < 3; ++i)
     cylinder.Color[i] = molColor[3][i];
  }
 }
 else {
  for(Cylinder &cylinder : bondCylinder[idx])
   for(int i = 0; i < 3; ++i)
    cylinder.Color[i] = atomColor[6][i];
 }
}

void Detrial_Unit::create_pept_triangle() {
 if(!peptTriangle.empty()) {  //作成済みなら何もしない
  return;
 }

 int mol_cnt = molVect.size();
 for(int i = 0; i < mol_cnt; ++i) {
  int pept_cnt = molVect[i].aaTable.size();
  for(int j = 0; j < pept_cnt; ++j) {
   int aa_cnt = molVect[i].aaTable[j].size();
   capture_terminal_triangle(i, j, aa_cnt);  //N-末と C-末の三角形
   if(aa_cnt > 2) {
    //3 個ずつ処理すると中央のアミノ酸部分の色が決定できる（末端は未処理）
    aa_cnt -= 2;
    for(int k = 0; k < aa_cnt; ++k) {
     capture_triangle(i, j, k, k + 1, k + 2);
    }
   }
  }
 }
}

void Detrial_Unit::capture_terminal_triangle(int m, int p, int aa_cnt) {
 const Amino_Acid &aa0 = molVect[m].aaTable[p][0];
 const Amino_Acid &aa1 = molVect[m].aaTable[p][1];
 int atomN = aa0.Annot.at(N);
 int atomCA =aa0.Annot.at(CA);
 int atomC = aa0.Annot.at(C);
 push_triangle(m, p, atomN, atomCA, atomC, aa0.Code1);
 atomN = aa1.Annot.at(N);
 push_triangle(m, p, atomCA, atomC, atomN, aa0.Code1);

 const Amino_Acid &aa2 = molVect[m].aaTable[p][aa_cnt - 2];
 const Amino_Acid &aa3 = molVect[m].aaTable[p][aa_cnt - 1];
 atomC = aa2.Annot.at(C);
 atomN = aa3.Annot.at(N);
 atomCA =aa3.Annot.at(CA);
 push_triangle(m, p, atomC, atomN, atomCA, aa3.Code1);
 atomC = aa3.Annot.at(C);
 push_triangle(m, p, atomN, atomCA, atomC, aa3.Code1);
}

void Detrial_Unit::capture_triangle(int m, int p, int a0, int a1, int a2) {
 const Amino_Acid &aa0 = molVect[m].aaTable[p][a0];
 Amino_Acid &aa1 = molVect[m].aaTable[p][a1];  //二次構造の色を設定する
 const Amino_Acid &aa2 = molVect[m].aaTable[p][a2];

 int atomC0 = aa0.Annot.at(C);
 int atomN1 = aa1.Annot.at(N);
 int atomCA1 = aa1.Annot.at(CA);
 push_triangle(m, p, atomC0, atomN1, atomCA1, aa1.Code1);
 int atomC1 = aa1.Annot.at(C);
 push_triangle(m, p, atomN1, atomCA1, atomC1, aa1.Code1);
 int atomN2 = aa2.Annot.at(N);
 push_triangle(m, p, atomCA1, atomC1, atomN2, aa1.Code1);

 double angle0 = measure_diheadral(atomC0, atomN1, atomCA1, atomC1);
 double angle1 = measure_diheadral(atomN1, atomCA1, atomC1, atomN2);
 int cnt = peptTriangle.size();
 Triangle &tri0 = peptTriangle[cnt - 3];
 Triangle &tri1 = peptTriangle[cnt - 2];
 Triangle &tri2 = peptTriangle[cnt - 1];
 if(0.5 < angle0 && angle0 < 2.5 && 0.0 < angle1 && angle1 < 1.0) {  //α-ヘリックス
  tri0.Secondary = 0;
  tri1.Secondary = 0;
  tri2.Secondary = 0;
  aa1.Secondary = 0;
 }
 else if(0.5 < angle0 && angle0 < 2.5 && -3.0 < angle1 && angle1 < -2.0) {  //β-シート
  tri0.Secondary = 2;
  tri1.Secondary = 2;
  tri2.Secondary = 2;
  aa1.Secondary = 2;
 }
}

void Detrial_Unit::push_triangle(int m, int p, int s0, int s1, int s2, int code) {
 Triangle tri;
 tri.molIdx = m;
 tri.peptIdx = p;
 tri.aaCode = code;
 tri.Serial0 = s0;
 tri.Pos0 = &atomVect[serialToIdx[s0]].XYZ;
 tri.Serial1 = s1;
 tri.Pos1 = &atomVect[serialToIdx[s1]].XYZ;
 tri.Serial2 = s2;
 tri.Pos2 = &atomVect[serialToIdx[s2]].XYZ;
 peptTriangle.push_back(tri);
}

void Detrial_Unit::set_pept_triangle_color(COLOR color) {
 if(color == SECONDARY) {
  for(Triangle &tri : peptTriangle) {
   if(tri.Secondary == 1) {
    tri.Color[0] = atomColor[0][0];
    tri.Color[1] = atomColor[0][1];
    tri.Color[2] = atomColor[0][2];
   }
   else {
    tri.Color[0] = molColor[tri.Secondary][0];
    tri.Color[1] = molColor[tri.Secondary][1];
    tri.Color[2] = molColor[tri.Secondary][2];
   }
  }
 }
 else if(color == PEPT) {
  int cl = -1;
  for(Triangle &tri : peptTriangle) {
   cl = (tri.molIdx * 10 + tri.peptIdx) % 6;
   tri.Color[0] = molColor[cl][0];
   tri.Color[1] = molColor[cl][1];
   tri.Color[2] = molColor[cl][2];
  }
 }
 else if(color == MONO) {
  for(Triangle &tri : peptTriangle) {
   tri.Color[0] = atomColor[0][0];
   tri.Color[1] = atomColor[0][1];
   tri.Color[2] = atomColor[0][2];
  }
 }
 else if(color == MOL) {
  int cl = -1;
  for(Triangle &tri : peptTriangle) {
   cl = (tri.molIdx) % 6;
   tri.Color[0] = molColor[cl][0];
   tri.Color[1] = molColor[cl][1];
   tri.Color[2] = molColor[cl][2];
  }
 }
 else if(color == AA) {
  int cl = -1;
  for(Triangle &tri : peptTriangle) {
   cl = -1;
   if(tri.aaCode == 'D' || tri.aaCode == 'E')
    cl = 0;
   else if(tri.aaCode == 'H' || tri.aaCode == 'K' || tri.aaCode == 'R')
    cl = 1;
   else if(tri.aaCode == 'C' || tri.aaCode == 'c')
    cl = 2;
   else if(tri.aaCode == 'S' || tri.aaCode == 'T' || tri.aaCode == 'N' || tri.aaCode == 'Q' || tri.aaCode == 'Y' || tri.aaCode == 'G')
    cl = 3;

   if(cl > -1)
    for(int i = 0; i < 3; ++i)
     tri.Color[i] = molColor[cl][i];
   else
    for(int i = 0; i < 3; ++i)
     tri.Color[i] = atomColor[0][i];
  }
 }
}

void Detrial_Unit::divide_sphere_and_cylinder_by_aa() {
 atomSphere[1] = atomSphere[0];

 //α 値の設定に．タンパク質かどうかの情報を使う．
 //初期化
 int cnt = atomSphere[0].size();
 for(int i = 0; i < cnt; ++i) {
  atomSphere[0][i].Color[3] = 0.0f;
  atomSphere[1][i].Color[3] = 1.0f;
 }

 //ペプチドと非ペプチドで α 値を逆転
 for(const Mol &mol : molVect)
  for(const std::vector<Amino_Acid> &pept : mol.aaTable)
   for(const Amino_Acid &aa : pept)
    for(int serial : aa.allSerial) {
     atomSphere[0][serialToIdx[serial]].Color[3] = 1.0f;
     atomSphere[1][serialToIdx[serial]].Color[3] = 0.0f;
    }

 bondCylinder[1] = bondCylinder[0];
 cnt = bondCylinder[0].size();
 for(int i = 0; i < cnt; ++i) {
  if(bondCylinder[0][i].peptIdx > -1) {
   bondCylinder[0][i].Color[3] = 1.0f;
   bondCylinder[1][i].Color[3] = 0.0f;
  }
  else {
   bondCylinder[0][i].Color[3] = 0.0f;
   bondCylinder[1][i].Color[3] = 1.0f;
  }
 }
}

void Detrial_Unit::divide_sphere_and_cylinder_by_mol_selection(const std::vector<int> *selected) {
 //初期化．すべて非選択とする
 atomSphere[1] = atomSphere[0];
 int atom_cnt = atomSphere[0].size();
 for(int i = 0; i < atom_cnt; ++i) {
  atomSphere[0][i].Color[3] = 1.0f;
  atomSphere[1][i].Color[3] = 0.0f;
 }
 bondCylinder[1] = bondCylinder[0];
 int bond_cnt = bondCylinder[0].size();
 for(int i = 0; i < bond_cnt; ++i) {
  bondCylinder[0][i].Color[3] = 1.0f;
  bondCylinder[1][i].Color[3] = 0.0f;
 }

 //分別．選択された分子を見つけて α 値を逆転する
 for(int i = 0; i < atom_cnt; ++i)
  if(binary_search(selected->begin(), selected->end(), atomVect[i].molIdx)) {
   atomSphere[0][i].Color[3] = 0.0f;
   atomSphere[1][i].Color[3] = 1.0f;
  }
 for(int i = 0; i < bond_cnt; ++i)
  if(binary_search(selected->begin(), selected->end(), bondCylinder[0][i].molIdx)) {
   bondCylinder[0][i].Color[3] = 0.0f;
   bondCylinder[1][i].Color[3] = 1.0f;
  }
}

void Detrial_Unit::divide_sphere_by_aa_selection(const std::vector<std::array<int, 3>> *selected) {
 //初期化．原子をすべて非選択とする
 atomSphere[1] = atomSphere[0];
 int atom_cnt = atomSphere[0].size();
 for(int i = 0; i < atom_cnt; ++i) {
  atomSphere[0][i].Color[3] = 1.0f;  //非選択側
  atomSphere[1][i].Color[3] = 0.0f;
 }
 //結合は，全て表示
 bondCylinder[1] = bondCylinder[0];
 int bond_cnt = bondCylinder[0].size();
 for(int i = 0; i < bond_cnt; ++i) {
  bondCylinder[0][i].Color[3] = 1.0f;
  bondCylinder[1][i].Color[3] = 1.0f;
 }

 //分別．選択されたアミノ酸を見つけて原子の α 値を逆転する
 for(const std::array<int, 3> &num : *selected) {
  const Amino_Acid &aa = molVect[num[0]].aaTable[num[1]][num[2]];
  for(int serial : aa.allSerial) {
   atomSphere[0][serialToIdx[serial]].Color[3] = 0.0f;
   atomSphere[1][serialToIdx[serial]].Color[3] = 1.0f;
  }
 }
}

void Detrial_Unit::clear_pick() {
 atomSphere[2].clear();
 atomSphere[2].shrink_to_fit();
}

const std::vector<int> *Detrial_Unit::picked_serial() {
 pickedSerial.clear();
 pickedSerial.shrink_to_fit();

 for(const Sphere &sphere : atomSphere[2])
  pickedSerial.push_back(sphere.Serial);

 return &pickedSerial;
}

const std::vector<std::array<int, 3>> *Detrial_Unit::pick_to_neighbor_aa(double range) {
 std::vector<int> picked;
 for(const Sphere &sphere : atomSphere[2])
  picked.push_back(sphere.Serial);
 return AtomNear.atom_to_neighbor_aa(&picked, range);
}

void Detrial_Unit::set_alpha_val(int idx, const std::vector<int> *val) {
 int atom_cnt = atomSphere[idx].size();
 for(int i = 0; i < atom_cnt; ++i)
  atomSphere[idx][i].Color[3] = (float)(*val)[i];
}

void Detrial_Unit::init_alpha_val(int idx) {
 int atom_cnt = atomSphere[idx].size();
 for(int i = 0; i < atom_cnt; ++i)
  atomSphere[idx][i].Color[3] = 1.0f;
}

bool Detrial_Unit::read_1_col_data(const std::string *file_name) {
 File_IO file;
 if(!file.read_1_col_data(file_name, &atomVect))
  return false;

 return true;
}
