#include "draw_base.h"

Draw_Base::Draw_Base(QWidget *parent,
        std::vector<Atom> *atom,
        const std::vector<Mol> *mol,
        const std::map<int, int> *serial,
        const Shared_Prm *prm)
 : QOpenGLWidget(parent),
   PressedButton(Qt::NoButton),
   SharedPrm(prm),
   AtomVect(atom),
   MolVect(mol),
   SerialToIdx(serial),
   QSurface(gluNewQuadric()),
   MouseX(0),
   MouseY(0),
   MouseMoveLength(0.0),
   Affine(atom, mol, serial),
   ZAxis(0.0, 0.0, 1.0) {
 Affine.calc_center();
 Affine.calc_radius();
 Affine.init_eye_pos();

 setMinimumWidth(600);
 setMinimumHeight(600);
}

void Draw_Base::initializeGL() {
 glClearColor(1.0, 1.0, 1.0, 1.0); //背景を白色に設定
 glEnable(GL_DEPTH_TEST);
 glEnable(GL_ALPHA_TEST);
}

void Draw_Base::resizeGL(int width, int height) {
 GLsizei length = (width > height) ? height : width;
 resize(length, length);  //描画領域を正方形に保つ
}

void Draw_Base::set_projection() {
 //射影変換
 glMatrixMode(GL_PROJECTION);
 glLoadIdentity();
 //field of view, aspect ratio, depth range
 gluPerspective(viewAngle,
        1.0,
        Affine.get_front(),
        Affine.get_back());
}

void Draw_Base::set_lookat() {
 const Eigen::Vector3d *eye_pos = Affine.get_eye_pos();
 const Eigen::Vector3d *center = Affine.get_draw_center();
 //モデルビュー変換
 glMatrixMode(GL_MODELVIEW);
 glLoadIdentity();
 //view point, focus point, up vector
 gluLookAt((*eye_pos)[0], (*eye_pos)[1], (*eye_pos)[2],
        (*center)[0], (*center)[1], (*center)[2],
        0.0, 1.0, 0.0);
}

void Draw_Base::paintGL() {
 glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

 set_projection();
 set_lookat();

 glEnable(GL_LIGHTING);
 glEnable(GL_LIGHT0);
 glEnable(GL_CULL_FACE);
 glCullFace(GL_BACK);

 gluQuadricDrawStyle(QSurface, GLU_FILL);
 draw_picked_sphere();  //ピックされた原子を先に描画しなくてはならない
 draw_sphere();
 draw_cylinder();
 draw_cone();

// gluQuadricDrawStyle(QSurface, GLU_LINE);

 glDisable(GL_LIGHTING);
 glDisable(GL_CULL_FACE);
 draw_triangle();
 draw_wire();
}

void Draw_Base::draw_sphere() {
 if(SphereForDraw.empty()) return;

 for(const Sphere &sphere : SphereForDraw) {
  glMaterialfv(GL_FRONT, GL_DIFFUSE, sphere.Color);
  glPushMatrix();
  glTranslated((*sphere.Pos)[0],
        (*sphere.Pos)[1],
        (*sphere.Pos)[2]);
  gluSphere(QSurface,
        sphere.Radius,
        Affine.get_slices(),
        Affine.get_stacks());
  glPopMatrix();
 }
}

void Draw_Base::draw_picked_sphere() {
 if(PickedSphereForDraw.empty()) return;
}

void Draw_Base::draw_cylinder() {
 if(CylinderForDraw.empty()) return;

 Eigen::Vector3d tmp_vect{0.0, 0.0, 0.0};
 Eigen::Vector3d tmp_axis{0.0, 0.0, 0.0};
 GLint slices = Affine.get_slices();
 for(const Cylinder &cylinder : CylinderForDraw) {
  tmp_vect = (*cylinder.Pos1) - (*cylinder.Pos0);
  double norm = tmp_vect.norm();
  tmp_vect = tmp_vect.normalized();
  //結合と Z 軸との成す角を内積 (z 軸 = 0.0, 0.0, 1.0) から求める
  double angle = acos(tmp_vect[2]) * RAD_to_DEG;
  //回転軸を外積から求める
  tmp_axis = zAxis.cross(tmp_vect).normalized();
  const Eigen::Vector3d &pos = (*cylinder.Pos0);
  glMaterialfv(GL_FRONT, GL_DIFFUSE, cylinder.Color);
  glPushMatrix();
  glTranslated(pos[0], pos[1], pos[2]);  //シリンダを平行移動
  glRotated(angle, tmp_axis[0], tmp_axis[1], tmp_axis[2]);  //同回転
  gluCylinder(QSurface, cylinder.Radius, cylinder.Radius, norm, slices, 1);
   glPopMatrix ();
 }
}

void Draw_Base::draw_cone() {
 if(ConeForDraw.empty()) return;

 Eigen::Vector3d tmp_vect{0.0, 0.0, 0.0};
 Eigen::Vector3d tmp_axis{0.0, 0.0, 0.0};
 GLint slices = Affine.get_slices();
 for(const Cone &cone : ConeForDraw) {
  tmp_vect = (*cone.Pos1) - (*cone.Pos0);
  double norm = tmp_vect.norm();
  tmp_vect = tmp_vect.normalized();
  //結合と Z 軸との成す角を内積 (z 軸 = 0.0, 0.0, 1.0) から求める
  double angle = acos(tmp_vect[2]) * RAD_to_DEG;
  //回転軸を外積から求める
  tmp_axis = zAxis.cross(tmp_vect).normalized();
  const Eigen::Vector3d &pos = (*cone.Pos0);
  glMaterialfv(GL_FRONT, GL_DIFFUSE, cone.Color);
  glPushMatrix();
  glTranslated(pos[0], pos[1], pos[2]);  //シリンダを平行移動
  glRotated(angle, tmp_axis[0], tmp_axis[1], tmp_axis[2]);  //同回転
  gluCylinder(QSurface, cone.Radius, 0.0, norm, slices, 1);
   glPopMatrix ();
 }
}

void Draw_Base::draw_triangle() {
 if(TriangleForDraw.empty()) return;
}

void Draw_Base::draw_wire() {
 if(WireForDraw.empty()) return;

 glBegin(GL_LINES);
 for(const Wire &wire : WireForDraw) {
  glColor3f(wire.Color[0], wire.Color[1], wire.Color[2]);
  glVertex3f((*wire.Pos0)[0], (*wire.Pos0)[1], (*wire.Pos0)[2]);
  glVertex3f((*wire.Pos1)[0], (*wire.Pos1)[1], (*wire.Pos1)[2]);
 }
 glEnd();
}

void Draw_Base::mousePressEvent(QMouseEvent *event) {
 if(!AtomVect || AtomVect->empty()) return;
 if(SharedPrm->KeyZ) return;

 //回転用の設定．描画領域は正方形に設定されている
 double draw_radius = Affine.get_draw_radius();
 MouseMoveLength = 2.0 * draw_radius / (double)width();
 PressedButton = event->button();
 MouseX = event->x();
 MouseY = event->y();
}

void Draw_Base::mouseMoveEvent(QMouseEvent *event) {
 if(!AtomVect || AtomVect->empty()) return;
 if(SharedPrm->KeyZ) return;

 if(SharedPrm->KeyX)  //視点を移動
  move_eye(event->x(), event->y());
 else if(PressedButton == Qt::LeftButton) {
  rotate_all_atom_xy(event->x(), event->y());
  rotate_other_object_xy(event->x(), event->y());
 }

 MouseX = event->x();
 MouseY = event->y();

 repaint();
}

void Draw_Base::mouseReleaseEvent(QMouseEvent *event) {
 if(!AtomVect || AtomVect->empty()) return;

 MouseX = event->x();
 MouseY = event->y();
}

void Draw_Base::wheelEvent(QWheelEvent *event) {
 if(!AtomVect || AtomVect->empty())
  return;

 if(!SharedPrm->KeyZ)
  return;

 double d = event->angleDelta().y();
 double draw_radius = Affine.get_draw_radius();
 d *= draw_radius;
 d /= 500.0;

 const Eigen::Vector3d *eye = Affine.get_eye_pos();
 if((*eye)[2] >= 0.0)
  Affine.move_eye(2, -d);
 else
  Affine.move_eye(2, d);
 Affine.move_front(-d);
 Affine.move_back(-d);
 repaint();
}

void Draw_Base::move_eye(int ox, int oy) {
 double r = Affine.get_draw_radius();
 double dx = (double)(MouseX - ox) / (double)width() * r;
 double dy = (double)(MouseY - oy) / (double)height() * r;
 Affine.move_draw_center(0, dx);
 Affine.move_draw_center(1, -dy);
 Affine.move_eye(0, dx);
 Affine.move_eye(1, -dy);
}

void Draw_Base::rotate_all_atom_xy(int ox, int oy) {
 //縦方向の動作で X 軸を，横方向の動作で Y 軸を回転
 double dx = (double)(ox - MouseX) / (double)height() * 360.0;
 double dy = (double)(oy - MouseY) / (double)width() * 360.0;
 Affine.rotate_atom_xy(dx, dy);
}

