#include "h_bond_draw.h"

H_Bond_Draw::H_Bond_Draw(QWidget *parent,
        std::vector<Atom> *atom,
        const std::vector<Mol> *mol,
        const std::map<int, int> *serial,
        const Shared_Prm *prm)
 : Draw_Base(parent, atom, mol, serial, prm),
   CovRadius(Atom_Distance().cov_radius()),
   Adj(atom, serial),
   AtomRadius(0.25),
   BondRadius(0.15),
   HBondRadius(0.1),
   LengthMax(0.0),
   AngleMax(0.0),
   AngleMin(0.0)
{}

void H_Bond_Draw::create_objects() {
 create_sphere();
 create_cylinder();
}

void H_Bond_Draw::detect() {
 DonorIdx.clear();
 DonorIdx.shrink_to_fit();
 AcceptorIdx.clear();
 AcceptorIdx.shrink_to_fit();
 HBondIdx.clear();
 HBondIdx.shrink_to_fit();

 int atom_cnt = AtomVect->size();
 for(int i = 0; i < atom_cnt; ++i) {
  if((*AtomVect)[i].Elmt == 7)
    check_n(i);
  if((*AtomVect)[i].Elmt == 8)
   check_o(i);
  if((*AtomVect)[i].Elmt == 16)
   check_s(i);
 }

 std::array<int, 2> row{-1, -1};
 for(const std::array<int, 2> &donor : DonorIdx) {
  row[0] = donor[1];
  for(const std::array<int, 2> &acceptor : AcceptorIdx) {
   if(donor[0] != acceptor[0]) {
    if(check_length(donor[1], acceptor[0])) {
     if(check_angle(donor[0], donor[1], acceptor[0])) {
      row[1] = acceptor[0];
      HBondIdx.push_back(row);

     }
    }
   }
  }
 }
}

bool H_Bond_Draw::check_length(int idx_d, int idx_a) {
 const Atom &atom_d = (*AtomVect)[idx_d];
 const Atom &atom_a = (*AtomVect)[idx_a];

 double d = (atom_a.XYZ - atom_d.XYZ).norm();
 if(d < LengthMax)
  return true;
 return false;
}

bool H_Bond_Draw::check_angle(int d0, int d1, int a0) {
 const Eigen::Vector3d d0p = (*AtomVect)[d0].XYZ;
 const Eigen::Vector3d d1p = (*AtomVect)[d1].XYZ;
 const Eigen::Vector3d a0p = (*AtomVect)[a0].XYZ;
 double angle = measure_angle(&d0p, &d1p, &a0p);
 angle *= RAD_to_DEG;

 if(angle < AngleMin)
  return false;
 if(angle > AngleMax)
  return false;

 return true;
}

void H_Bond_Draw::check_n(int idx) {
 const Atom &atom = (*AtomVect)[idx];

 //アンモニウムイオンは除外
 int bond_cnt = atom.Bond.size();
 if(bond_cnt > 3)
  return;

 //イミダゾール基は除外
 if(bond_cnt == 3 && imidazole_n(idx))
  return;

 std::array<int, 2> nx;
 nx[0] = idx;
 std::array<int, 2> nh;
 nh[0] = idx;

 for(int serial_pair : atom.Bond) {
  int idx_pair = SerialToIdx->at(serial_pair);
  nx[1] = idx_pair;
  AcceptorIdx.push_back(nx);

  if((*AtomVect)[idx_pair].Elmt == 1) {
  nh[1] = idx_pair;
  DonorIdx.push_back(nh);
  }
 }
}

bool H_Bond_Draw::imidazole_n(int idx) {
 Adj.check((*AtomVect)[idx].serialNum);
 if(Adj.H.size() != 1)
  return false;
 if(Adj.C.size() != 2)
  return false;

 int idx0 = SerialToIdx->at(Adj.C[0]);
 const Atom &atom0 = (*AtomVect)[idx0];
 if(atom0.Bond.size() != 3)
  return false;

 int idx1 = SerialToIdx->at(Adj.C[1]);
 const Atom &atom1 = (*AtomVect)[idx1];
 if(atom1.Bond.size() != 3)
  return false;

 return true;
}

void H_Bond_Draw::check_o(int idx) {
 const Atom &atom = (*AtomVect)[idx];
 //オキソニウムイオンは除外
 if(atom.Bond.size() > 2)
  return;

 std::array<int, 2> ox;
 ox[0] = idx;
 std::array<int, 2> oh;
 oh[0] = idx;

 for(int serial_pair : atom.Bond) {
  int idx_pair = SerialToIdx->at(serial_pair);
  ox[1] = idx_pair;
  AcceptorIdx.push_back(ox);

  if((*AtomVect)[idx_pair].Elmt == 1) {
  oh[1] = idx_pair;
  DonorIdx.push_back(oh);
  }
 }
}

void H_Bond_Draw::check_s(int idx) {
 const Atom &atom = (*AtomVect)[idx];

 //硫酸は除外
 if(atom.Bond.size() > 3)
  return;

 std::array<int, 2> sx;
 sx[0] = idx;
 std::array<int, 2> sh;
 sh[0] = idx;

 for(int serial_pair : atom.Bond) {
  int idx_pair = SerialToIdx->at(serial_pair);
  sx[1] = idx_pair;
  AcceptorIdx.push_back(sx);

  if((*AtomVect)[idx_pair].Elmt == 1) {
  sh[1] = idx_pair;
  DonorIdx.push_back(sh);
  }
 }
}

void H_Bond_Draw::create_sphere() {
 SphereForDraw.clear();
 SphereForDraw.shrink_to_fit();

 Sphere sphere;

 for(const Atom &atom : *AtomVect) {
  sphere.Serial = atom.serialNum;
  sphere.Pos = &atom.XYZ;
  sphere.Radius = AtomRadius;
   for(int j = 0; j < 4; ++j)
    sphere.Color[j] = atomColor[atom.Elmt][j];
  SphereForDraw.push_back(sphere);
 }
}

void H_Bond_Draw::create_cylinder() {
 CylinderForDraw.clear();
 CylinderForDraw.shrink_to_fit();

 Cylinder cylinder;
 cylinder.Radius = BondRadius;

 for(const Atom &atom : *AtomVect) {
  cylinder.Pos0 = &atom.XYZ;
  for(int serial1 : atom.BondUp) {
   int idx1 = SerialToIdx->at(serial1);
   const Atom &atom1 = (*AtomVect)[idx1];
   cylinder.Pos1 = &(atom1.XYZ);
   CylinderForDraw.push_back(cylinder);
  }
 }

 cylinder.Radius = HBondRadius;
 cylinder.Color[0] = 0.0;
 cylinder.Color[1] = 1.0;
 cylinder.Color[2] = 0.0;
 cylinder.Color[3] = 1.0;
 for(const std::array<int, 2> &row : HBondIdx) {
  cylinder.Pos0 = &(*AtomVect)[row[0]].XYZ;
  cylinder.Pos1 = &(*AtomVect)[row[1]].XYZ;
  CylinderForDraw.push_back(cylinder);
 }
}

