#include "plot_window.h"
#include <builcule/libbuilcule_global.h>
#include <QBoxLayout>
#include <QDesktopWidget>
#include <QMessageBox>
#include <QPainter>
#include <QScrollArea>
#include <cmath>

Plot_Window::Plot_Window(QWidget *parent)
 : QMainWindow(parent),
   Threshold(0.65)
{
 //ウィンドウを表示しているあいだは，メインウィンドウでの操作を禁止する
 setWindowModality(Qt::WindowModal);
 setWindowFlags(windowFlags() | Qt::WindowStaysOnTopHint);

 QWidget *widget_tool = new QWidget(this);
 widget_tool->setFixedSize(300, 50);
 QHBoxLayout *layout_tool = new QHBoxLayout(widget_tool);

 leThres = new QLineEdit(QString::number(Threshold), widget_tool);
 leThres->setMaxLength(5);

 pbReplot = new QPushButton("再描画", widget_tool);
 connect(pbReplot, SIGNAL(clicked()), this, SLOT(slot_replot()));

 pbClose = new QPushButton("閉じる", widget_tool);
 connect(pbClose, SIGNAL(clicked()), this, SLOT(slot_close()));

 layout_tool->addWidget(leThres);
 layout_tool->addWidget(pbReplot);
 layout_tool->addWidget(pbClose);

 plotWidget = new Plot_Widget(this);
 plotWidget->set_score_mtx(&scoreMtx);
 plotWidget->set_level(&Level);
 plotWidget->set_threshold(&Threshold);

 //レイアウト
 QWidget *widget_central = new QWidget(this);
 setCentralWidget(widget_central);
 QVBoxLayout *layout_central = new QVBoxLayout(widget_central);
 layout_central->addWidget(widget_tool);

 //描画領域．スクロールバーを付ける
 QScrollArea *scroll_area = new QScrollArea(this);
 scroll_area->setWidget(plotWidget);
 layout_central->addWidget(scroll_area);

 calc_level();
}

void Plot_Window::calc_level() {
 double range = (1.0 - Threshold) / 5.0;
 for(int i = 1; i < 5; ++i)
  Level[i - 1] = Threshold + range * (double)i;
}

void Plot_Window::set_score_mtx(const std::vector<std::vector<double>> *mtx) {
 if(mtx->empty()) {
  std::cerr << __PRETTY_FUNCTION__ << std::endl;
  return;
 }

 int row_cnt = mtx->size() - 1;
 scoreMtx.assign(row_cnt, std::vector<double>());

 for(int i = 0; i != row_cnt; ++i) {
  scoreMtx[i].insert(scoreMtx[i].end(), (*mtx)[i + 1].begin() + 1, (*mtx)[i + 1].end());
 }

 //スケーリング
 double max = score_max();
 double min = score_min();
 double diff = max - min;
 if(fabs(diff) < EPSILON) {
  QMessageBox::warning(this, "警告", "スコアの最大値と最小値がほぼ同じ値です．");
  return;
 }

 for(std::vector<double> &row : scoreMtx)
  for(double &val : row) {
   val = (val - min) / diff;
  }
}

void Plot_Window::set_window_size() {
 if(scoreMtx.empty())
  return;

 QDesktopWidget screen;
 int s_width = screen.width();
 int s_height = screen.height();

 int d_width = scoreMtx[0].size() +150;
 if(d_width < 300)
  d_width = 300;
 int d_height = scoreMtx.size() + 200;

 if(s_width < d_width || s_height < d_height)
  setWindowState(windowState() ^ Qt::WindowMaximized);
 else
  setFixedSize(d_width, d_height);
}

double Plot_Window::score_max() {
 double max_val = scoreMtx[0][0];
 for(const std::vector<double> &row : scoreMtx) {
  for(const double val : row) {
   if(max_val < val) max_val = val;
  }
 }
 return max_val;
}

double Plot_Window::score_min() {
 double min_val = scoreMtx[0][0];
 for(const std::vector<double> &row : scoreMtx) {
  for(const double val : row) {
   if(min_val > val) min_val = val;
  }
 }
 return min_val;
}

Plot_Widget::Plot_Widget(QWidget *parent)
 : QWidget(parent),
   margin(50)
{
 setMinimumSize(1, 1);  //何らかの値を設定しておかないと描画されない
 setStyleSheet("background-color:white");

 Color[0] = Qt::blue;
 Color[1] = Qt::cyan;
 Color[2] = Qt::green;
 Color[3] = Qt::yellow;
 Color[4] = Qt::red;
}

void Plot_Window::slot_replot() {
 QString str = leThres->text();
 if(str.isEmpty()) {
  QMessageBox::critical(this, "エラー", "閾値が空欄です．");
  return;
 }
 double thres = str.toDouble();
 if(thres < 0.0 || 1.0 < thres) {
  QMessageBox::critical(this, "エラー", "ウェイトが異常値です．");
  return;
 }

 Threshold = thres;
 calc_level();
 plotWidget->plot();
}

void Plot_Window::slot_close() {
 close();
}

void Plot_Widget::paintEvent(QPaintEvent *) {
 if(scoreMtx->empty())
  return;

 draw_frame();
 draw_dot();
}

void Plot_Widget::draw_frame() {
 int x0 = margin;
 int y0 = margin;
 int y_width = scoreMtx->size();
 int x_width = (*scoreMtx)[0].size();
 resize(x_width + margin * 2, y_width + margin * 2);

 QPainter painter(this);

 //枠
 painter.drawRect(x0, y0, x_width, y_width);

 //目盛
 int scale = 100;
 int scale_cnt = x_width / scale;  //100 ドットごとに目盛を打つ
 for(int i = 1; i <= scale_cnt; ++i) {
  painter.drawLine(x0 + scale * i, y0, x0 + scale * i, y0 - 10);
  painter.drawLine(x0 + scale * i, y0 + y_width, x0 + scale * i, y0 + y_width + 10);
 }

 scale_cnt = y_width / scale;
 for(int i = 1; i <= scale_cnt; ++i) {
  painter.drawLine(x0, y0 + scale * i, x0 - 10, y0 + scale * i);
  painter.drawLine(x0 + x_width, y0 + scale * i, x0 + x_width + 10, y0 + scale * i);
 }
}

void Plot_Widget::draw_dot() {
 int x0 = margin;
 int y0 = margin;
 int x_pos = 0;
 int y_pos = 0;
 QPainter painter(this);
 int y_cnt = scoreMtx->size();
 int x_cnt = (*scoreMtx)[0].size();
 for(int i = 0; i < y_cnt; ++i) {
  y_pos = y0 + i;
  for(int j = 0; j < x_cnt; ++j) {
   double score = (*scoreMtx)[i][j];
   x_pos = x0 + j;
   if(score >= (*Level)[3]) {
    painter.setPen(Color[4]);
    painter.drawPoint(x_pos, y_pos);
   }
   else if(score >= (*Level)[2]) {
    painter.setPen(Color[3]);
    painter.drawPoint(x_pos, y_pos);
   }
   else if(score >= (*Level)[1]) {
    painter.setPen(Color[2]);
    painter.drawPoint(x_pos, y_pos);
   }
   else if(score >= (*Level)[0]) {
    painter.setPen(Color[1]);
    painter.drawPoint(x_pos, y_pos);
   }
   else if(score >= *Threshold) {
    painter.setPen(Color[0]);
    painter.drawPoint(x_pos, y_pos);
   }
  }
 }
}

