#include "vdw_contact_draw.h"

VDW_Contact_Draw::VDW_Contact_Draw(QWidget *parent,
        std::vector<Atom> *atom,
        const std::vector<Mol> *mol,
        const std::map<int, int> *serial,
        const Shared_Prm *prm)
 : Draw_Base(parent, atom, mol, serial, prm),
   Thres(0.0)
{}

void VDW_Contact_Draw::create_objects() {
 //チェックテーブルを作成
 int atom_cnt = AtomVect->size();
 VdwContact.assign(atom_cnt, std::vector<bool>(atom_cnt, true));

 //結合した原子，1 原子を介して結合した原子を除外
 for(int i = 0; i < atom_cnt; ++i)
  exclude_bound_pair(i, VdwContact[i]);

 //リストアップされたペアについて距離をチェック
 exclude_far_pair();

 //針金 + 空間充填モデル
 create_sphere();
 create_wire();
}

void VDW_Contact_Draw::exclude_bound_pair(int idx0, std::vector<bool> &row) {
 const Atom &atom0 = (*AtomVect)[idx0];
 const std::vector<int> &bond0 = atom0.Bond;
 for(int pair1 : bond0) {
  int idx1 = SerialToIdx->at(pair1);
  row[idx1] = false;

  const Atom &atom1 = (*AtomVect)[idx1];
  const std::vector<int> &bond1 = atom1.Bond;
  for(int pair2 : bond1) {
   int idx2 = SerialToIdx->at(pair2);
   row[idx2] = false;
  }
 }
}

void VDW_Contact_Draw::exclude_far_pair() {
 Atom_Distance distance;
 const std::array<std::array<double, ELMT_CNT>, ELMT_CNT> &vdw_lim
        = distance.vdw_contact();
 int atom_cnt = AtomVect->size();
 int i_end = atom_cnt - 1;
 for(int i = 0; i < i_end; ++i) {
  const Eigen::Vector3d &xyz0 = (*AtomVect)[i].XYZ;
  const int elmt0 = (*AtomVect)[i].Elmt;
  for(int j = i + 1; j < atom_cnt; ++j) {
   const Eigen::Vector3d &xyz1 = (*AtomVect)[j].XYZ;
   const int elmt1 = (*AtomVect)[j].Elmt;
   if((xyz1 - xyz0).norm() > vdw_lim[elmt0][elmt1] * Thres)
    VdwContact[i][j] = false;
  }
 }
}

void VDW_Contact_Draw::create_sphere() {
 SphereForDraw.clear();
 SphereForDraw.shrink_to_fit();

 //球に描くべき原子をリストアップ
 int atom_cnt = AtomVect->size();
 std::vector<bool> atom_obj(atom_cnt, false);
 int i_end = atom_cnt - 1;
 for(int i = 0; i < i_end; ++i) {
  for(int j = i + 1; j < atom_cnt; ++j) {
   if(VdwContact[i][j]) {
    atom_obj[i] = true;
    atom_obj[j] = true;
   }
  }
 }

 //球をプッシュ
 Atom_Distance distance;
 const std::array<double, ELMT_CNT> &VdwRadius
        = distance.vdw_radius();
 Sphere sphere;
 for(int i = 0; i < atom_cnt; ++i)
  if(atom_obj[i]) {
   const Atom &atom = (*AtomVect)[i];
   sphere.Serial = atom.serialNum;
   sphere.Pos = &atom.XYZ;
   sphere.Radius = VdwRadius[atom.Elmt];
   for(int j = 0; j != 4; ++j)
    sphere.Color[j] = atomColor[atom.Elmt][j];
   SphereForDraw.push_back(sphere);
  }
}

void VDW_Contact_Draw::create_wire() {
 WireForDraw.clear();
 WireForDraw.shrink_to_fit();

 int color_num = 0;
 Wire wire;
 for(const Mol &mol : *MolVect) {
  const std::vector<int> &serial_vect = mol.atomSerial;
  for(int serial0 : serial_vect) {
   int idx0 = SerialToIdx->at(serial0);
   const Atom &atom0 = (*AtomVect)[idx0];
   const std::vector<int> &bond = atom0.BondUp;
   for(int serial1 : bond) {
    int idx1 = SerialToIdx->at(serial1);
    wire.Pos0 = &(*AtomVect)[idx0].XYZ;
    wire.Pos1 = &(*AtomVect)[idx1].XYZ;
    for(int j = 0; j != 3; ++j)
     wire.Color[j] = molColor[color_num][j];

    WireForDraw.push_back(wire);
   }
  }
  color_num = (color_num + 1)  % 6;
 }
}

