#include "affine_trans_2.h"

Affine_Trans_2::Affine_Trans_2(std::vector<Atom> *atom, const std::vector<Mol> *mol, const std::map<int, int> *serial)
 : AtomVect(atom),
   MolVect(mol),
   SerialToIdx(serial),
   Center(Eigen::Vector3d::Zero(3)),
   DrawCenter(Eigen::Vector3d::Zero(3)),
   Radius(0.0),
   DrawRadius(0.0),
   Slices(0),
   Stacks(0),
   EyePos(Eigen::Vector3d::Zero(3)),
   Front(0.0),
   Back(0.0)
{
 //XY 平面上の回転行列
 double factor = 360.0 / M_PI;
 Eigen::Vector3d unit_x = Eigen::Vector3d::UnitX();
 Eigen::Vector3d unit_y = Eigen::Vector3d::UnitY();
 for(int i = -180; i != 181; ++i)
  for(int j = -180; j != 181; ++j)
   RotMtxXY[i + 180][j + 180] =
        Eigen::AngleAxisd((double)j / factor, unit_x)
        * Eigen::AngleAxisd((double)i / factor, unit_y);
}

void Affine_Trans_2::calc_center() {
 Center = (*AtomVect)[0].XYZ;
 int atom_cnt = AtomVect->size();
 for(int i = 1; i < atom_cnt; ++i)
  Center += (*AtomVect)[i].XYZ;
 Center /= (double)atom_cnt;
 DrawCenter = Center;
}

void Affine_Trans_2::calc_radius() {
 Radius = 0.0;
 double d = 0.0;
 for(const Atom &atom : *AtomVect) {
  d = (atom.XYZ - Center).norm();
  if(Radius < d)
   Radius = d;
 }
 DrawRadius = Radius + 2.75;
}

void Affine_Trans_2::init_eye_pos() {
 DrawCenter = Center;
 EyePos[0] = DrawCenter[0];
 EyePos[1] = DrawCenter[1];
 EyePos[2] = DrawCenter[2] + DrawRadius * 10.0;

 Front = DrawRadius * 9.0;
 Back = DrawRadius * 11.0;

 if(Front < 10.0)
  Slices = 64;
 else if(Front < 100.0)
  Slices = 32;
 else
  Slices = 16;

 Stacks = Slices / 2;
}

void Affine_Trans_2::rotate_atom_xy(double dx, double dy) {
 if(dx < -180.0 || 180.0 < dx) return;
 if(dy < -180.0 || 180.0 < dy) return;

 for(Atom &atom : *AtomVect) {
  atom.XYZ -= DrawCenter;
  atom.XYZ = RotMtxXY[(int)dx+ 180][(int)dy + 180] * atom.XYZ;
  atom.XYZ += DrawCenter;
 }
}

void Affine_Trans_2::rotate_point_xy(Eigen::Vector3d *point, double dx, double dy) {
 if(dx < -180.0 || 180.0 < dx) return;
 if(dy < -180.0 || 180.0 < dy) return;

 *point -= DrawCenter;
 *point = RotMtxXY[(int)dx+ 180][(int)dy + 180] * (*point);
 *point += DrawCenter;
}

void Affine_Trans_2::resize_mol() {
 int mol_cnt = MolVect->size();
 MolCenter.resize(mol_cnt, Eigen::Vector3d::Zero());
 MolRadius.resize(mol_cnt, 0.0);
}

void Affine_Trans_2::calc_mol_center() {
 int mol_cnt = MolVect->size();
 MolCenter.assign(mol_cnt, Eigen::Vector3d::Zero());
 for(int i = 0; i < mol_cnt; ++i)
  calc_mol_center(i);
}

void Affine_Trans_2::calc_mol_center(int mol) {
 if((int)MolCenter.size() < mol) {
  std::cerr << __PRETTY_FUNCTION__ << std::endl;
  return;
 }

 const std::vector<int> &atom_serial = (*MolVect)[mol].atomSerial;
 int idx0 = SerialToIdx->at(atom_serial[0]);
 MolCenter[mol] = (*AtomVect)[idx0].XYZ;
 int atom_cnt = atom_serial.size();
 for(int i = 1; i < atom_cnt; ++i) {
  int idx = SerialToIdx->at(atom_serial[i]);
  MolCenter[mol] += (*AtomVect)[idx].XYZ;
 }
 MolCenter[mol] /= (double)atom_cnt;
}

void Affine_Trans_2::calc_mol_radius() {
 int mol_cnt = MolVect->size();
 MolRadius.assign(mol_cnt, 0.0);
 for(int i = 0; i < mol_cnt; ++i)
  calc_mol_radius(i);
}

void Affine_Trans_2::calc_mol_radius(int mol) {
 if((int)MolRadius.size() < mol) {
  std::cerr << __PRETTY_FUNCTION__ << std::endl;
  return;
 }

 MolRadius[mol] = 0.0;
 const std::vector<int> &atom_serial = (*MolVect)[mol].atomSerial;
 int atom_cnt = atom_serial.size();
 for(int i = 0; i < atom_cnt; ++i) {
  int idx = SerialToIdx->at(atom_serial[i]);
  double r = ((*AtomVect)[idx].XYZ - MolCenter[mol]).norm();
  if(MolRadius[mol] < r)
   MolRadius[mol] = r;
 }
}

const Eigen::Vector3d *Affine_Trans_2::get_mol_center(int mol) const {
 if((int)MolCenter.size() < mol) {
  std::cerr << __PRETTY_FUNCTION__ << std::endl;
  return nullptr;
 }

 return &MolCenter[mol];
}

double Affine_Trans_2::get_mol_radius (int mol) const {
 if((int)MolRadius.size() < mol) {
  std::cerr << __PRETTY_FUNCTION__ << std::endl;
  return 0.0;
 }

 return MolRadius[mol];
}

void Affine_Trans_2::move_mol(int mol, const Eigen::Vector3d *v) {
 const std::vector<int> &atom_serial = (*MolVect)[mol].atomSerial;
 for(int serial : atom_serial) {
  int idx = SerialToIdx->at(serial);
  (*AtomVect)[idx].XYZ += *v;
 }

 if((int)MolCenter.size() < mol) {
  std::cerr << __PRETTY_FUNCTION__ << std::endl;
  return;
 }
 MolCenter[mol] += *v;
}
