#include "charge.h"

Charge::Charge(
        std::vector<Atom> *atom_vect,
        std::map<int, int> *idx_map,
        std::vector<Mol> *mol_vect)
 : AtomVect(atom_vect),
   SerialToIdx(idx_map),
   MolVect(mol_vect),
   AdjAtom(atom_vect, idx_map)
{}

void Charge::calc_charge() {
 //初期化
 for(Atom &atom : *AtomVect)
  atom.Charge = 0.0;

 //計算
 for(Atom &atom : *AtomVect) {
  //単原子イオンまたは金属が生体構成分子に配位結合している場合を想定
  if(atom.Symbol == "H" && atom.Bond.empty())
   atom.Charge += 1.0;
  else if(atom.Symbol == "Na" && atom.Bond.empty())
   atom.Charge += 1.0;
  else if(atom.Symbol == "K" && atom.Bond.empty())
   atom.Charge += 1.0;
  else if(atom.Symbol == "Cl" && atom.Bond.empty())
   atom.Charge += -1.0;
  else if(atom.Symbol == "Mg" && atom.Bond.empty())
   atom.Charge += 2.0;
  else if(atom.Symbol == "Ca" && atom.Bond.empty())
   atom.Charge += 2.0;
  else if(atom.Symbol == "Mn")
   atom.Charge += 2.0;
  else if(atom.Symbol == "Fe")
   atom.Charge += 2.0;
  else if(atom.Symbol == "Co")
   atom.Charge += 1.0;
  else if(atom.Symbol == "Cu")
   atom.Charge += 1.0;
  //官能基
  else if(atom.Elmt == 6 && atom.Orbital == SP2)
   check_CO_charge(atom);
  else if(atom.Elmt == 7)
   check_ammonium(atom);
  else if(atom.Elmt == 8)
   check_hydroxyl(atom);
  else if(atom.Elmt == 15)
   check_phosphate(atom);
  else if(atom.Elmt == 16)
   check_sulfate(atom);
 }
 //アミノ酸残基
 for(const Mol &mol : *MolVect) {
  check_his_arg(&mol);
 }
}

void Charge::check_his_arg(const Mol *mol) {
 for(const std::vector<Amino_Acid> &pept : mol->aaTable) {
  for(const Amino_Acid &aa : pept) {
   //His のイミダゾール基（プロトン化している場合）
   if(aa.Code1 == 'H') {
    int serial = aa.Annot.at(NE2);
    int idx = SerialToIdx->at(serial);
    if((*AtomVect)[idx].Bond.size() == 3)
     (*AtomVect)[idx].Charge += 1.0;
   }
   //Arg のグアニジウム基（プロトン化している場合）
   else if(aa.Code1 == 'R') {
    int serial = aa.Annot.at(NH1);
    int idx = SerialToIdx->at(serial);
    if((*AtomVect)[idx].Bond.size() == 3)
     (*AtomVect)[idx].Charge += 1.0;
   }
  }
 }
}

void Charge::check_CO_charge(Atom &atom) {
 AdjAtom.check(atom.serialNum);
 if(!AdjAtom.S.empty()) return;
 if(!AdjAtom.X.empty()) return;

 //ケトン/アルデヒド
 if((AdjAtom.C.size() == 2 && AdjAtom.O.size() == 1)
    || (AdjAtom.C.size() == 1 && AdjAtom.O.size() == 1 && AdjAtom.H.size() == 1)) {
  int serial0 = AdjAtom.O[0];
  int idx0 = SerialToIdx->at(serial0);
  Atom &atom0 = (*AtomVect)[idx0];
  if(atom0.Orbital != SP2)
   return;
  atom.Charge = 0.3;
  atom0.Charge = -0.3;
 }
 //ペプチド
 else if(AdjAtom.C.size() == 1 && AdjAtom.O.size() == 1 && AdjAtom.N.size() == 1) {
  int serial0 = AdjAtom.O[0];
  int idx0 = SerialToIdx->at(serial0);
  Atom &atom0 = (*AtomVect)[idx0];
  atom.Charge = 0.3;
  atom0.Charge = -0.3;
 }
 //エステル/カルボン酸
 else if((AdjAtom.C.size() == 1 && AdjAtom.O.size() == 2)
        || (AdjAtom.H.size() == 1 && AdjAtom.O.size() == 2)) {
  int serial0 = AdjAtom.O[0];
  int serial1 = AdjAtom.O[1];
  int idx0 = SerialToIdx->at(serial0);
  int idx1 = SerialToIdx->at(serial1);
  Atom &atom0 = (*AtomVect)[idx0];
  Atom &atom1 = (*AtomVect)[idx1];
  //カルボン酸（解離型）
  if(atom0.Bond.size() == 1 && atom1.Bond.size() == 1) {
   atom.Charge -= 1.0;
  }
  //エステル/カルボン酸（非解離型）
  else if(atom0.Bond.size() == 1 && atom1.Bond.size() == 2) {
   atom.Charge = 0.3;
   atom0.Charge = -0.3;
  }
  else if(atom0.Bond.size() == 2 && atom1.Bond.size() == 1) {
   atom.Charge = 0.3;
   atom1.Charge = -0.3;
  }
 }
}

void Charge::check_ammonium(Atom &atom) {
 if(atom.Orbital == SP3) {
  AdjAtom.check(atom.serialNum);
  if(AdjAtom.H.size() + AdjAtom.C.size() == 4)
   atom.Charge += 1.0;
 }
 else if(atom.Orbital == SP2) {  //核酸塩基の C-N-C 部分
  if(atom.Bond.size() != 2)
   return;
  int idx0 = SerialToIdx->at(atom.Bond[0]);
  int idx1 = SerialToIdx->at(atom.Bond[1]);
  Atom &atom0 = (*AtomVect)[idx0];
  Atom &atom1 = (*AtomVect)[idx1];
  if(atom0.Elmt != 6 && atom0.Orbital != SP2)
   return;
  if(atom1.Elmt != 6 && atom1.Orbital != SP2)
   return;
  atom.Charge = -0.1;
  atom0.Charge = 0.05;
  atom1.Charge = 0.05;
 }
}

void Charge::check_hydroxyl(Atom &atom) {
 if(atom.Bond.size() != 1)
  return;
 int serial = atom.Bond[0];
 int idx = SerialToIdx->at(serial);
 if((*AtomVect)[idx].Elmt != 1)
  return;

 atom.Orbital = SP3;
 atom.Charge -= 1.0;
}

void Charge::check_phosphate(Atom &atom) {
 if(atom.Bond.size() != 4) return;
 AdjAtom.check(atom.serialNum);
 if(AdjAtom.O.size() != 4) return;

 std::vector<int> free_o;
 for(int serial : AdjAtom.O) {
  int idx = SerialToIdx->at(serial);
  if((*AtomVect)[idx].Bond.size() == 1)
   free_o.push_back(idx);
 }
 int free_o_cnt = free_o.size();
 for(int i = 1; i < free_o_cnt; ++i)
  (*AtomVect)[free_o[i]].Charge += -1.0;
 atom.Orbital = SP3;
}

void Charge::check_sulfate(Atom &atom) {
 if(atom.Bond.size() == 4) {  //H2SO4 を想定
  AdjAtom.check(atom.serialNum);
  if(AdjAtom.O.size() != 4) return;

  std::vector<int> free_o;
  for(int serial : AdjAtom.O) {
   int idx = SerialToIdx->at(serial);
   if((*AtomVect)[idx].Bond.size() == 1)
    free_o.push_back(idx);
  }
  int free_o_cnt = free_o.size();
  for(int i = 2; i < free_o_cnt; ++i)
   (*AtomVect)[free_o[i]].Charge += -1.0;
  atom.Orbital = SP3;
 }
 else if(atom.Bond.size() == 2) { //CYS-SH を想定
  int idx0 = SerialToIdx->at(atom.Bond[0]);
  int idx1 = SerialToIdx->at(atom.Bond[1]);
  Atom &atom0 = (*AtomVect)[idx0];
  Atom &atom1 = (*AtomVect)[idx1];
  if(atom0.Elmt == 6 && atom1.Elmt == 1) {
   atom.Orbital = SP3;
   atom.Charge = -0.2;
   atom1.Charge = 0.2;
  }
  if(atom0.Elmt == 1 && atom1.Elmt == 6) {
   atom.Orbital = SP3;
   atom.Charge = -0.2;
   atom0.Charge = 0.2;
  }
 }
}

