#include "dynamic.h"
#include <algorithm>

Dynamic_Programming::Dynamic_Programming()
 : GapIni(-1.0),
   GapElg(-1.0),
   TBter0(-1),
   TBter1(-1)
{}

void Dynamic_Programming::clear()
{
 GapIni = -1.0;
 GapElg = -1.0;
 TBter0 = -1;
 TBter1 = -1;

 ScoreMtx.clear();
 ScoreMtx.shrink_to_fit();
 UpMtx.clear();
 UpMtx.shrink_to_fit();
 LeftMtx.clear();
 LeftMtx.shrink_to_fit();
 TraceMtx.clear();
 TraceMtx.shrink_to_fit();
 TBPoint.clear();
 TBPoint.shrink_to_fit();
 for(int i = 0; i < 2; ++i) {
  AlignedNum[i].clear();
  AlignedNum[i].shrink_to_fit();
 }
}

void Dynamic_Programming::init_mtx(int row, int col, double ini, double elg) {
 ScoreMtx.assign(row, std::vector<double>(col, 0.0));
 UpMtx.assign(row, std::vector<double>(col, 0.0));
 LeftMtx.assign(row, std::vector<double>(col, 0.0));
 TraceMtx.assign(row, std::vector<char>(col, '-'));

 GapIni = ini;
 GapElg = elg;

 ScoreMtx[0][0] = -ini;
 UpMtx[0][0] = -ini;
 LeftMtx[0][0] = -ini;

 for(int i = 1; i < col; ++i) {
  ScoreMtx[0][i] = ScoreMtx[0][i - 1] - elg;
  UpMtx[0][i] = UpMtx[0][i - 1] - elg;
  LeftMtx[0][i] = LeftMtx[0][i - 1] - elg;
 }

 for(int i = 1; i < row; ++i) {
  ScoreMtx[i][0] = ScoreMtx[i - 1][0] - elg;
  UpMtx[i][0] = UpMtx[i - 1][0] - elg;
  LeftMtx[i][0] = LeftMtx[i - 1][0] - elg;
 }

 for(int i = 1; i < row; ++i)
  TraceMtx[i][0] = '|';
}

void Dynamic_Programming::trace_1_step(int row, int col, double val)
{
 double score_d = 0.0;
 double score_l = 0.0;
 double score_u = 0.0;

 score_d = ScoreMtx[row - 1][col - 1] + val;

 double score_l0 = LeftMtx[row][col - 1] - GapElg;
 double score_l1 = ScoreMtx[row][col - 1] - GapIni;
 score_l = (score_l0 > score_l1) ? score_l0 : score_l1;

 double score_u0 = UpMtx[row - 1][col] - GapElg;
 double score_u1 = ScoreMtx[row - 1][col] - GapIni;
 score_u = (score_u0 > score_u1) ? score_u0 : score_u1;

 if(score_d > score_l && score_d > score_u && score_d > 0.0) {
  ScoreMtx[row][col] = score_d;
  TraceMtx[row][col] = 'X';
 }
 else if(score_l > score_d && score_l > score_u && score_l > 0.0) {
  ScoreMtx[row][col] = LeftMtx[row][col] = score_l;
  TraceMtx[row][col] = '-';
 }
 else if(score_u > score_d && score_u > score_l && score_u > 0.0)
  ScoreMtx[row][col] = UpMtx[row][col] = score_u;
}

const std::vector<TB_Point> *Dynamic_Programming::search_traceback_point() {
 int rter = ScoreMtx.size() - 1;
 int cter = ScoreMtx[0].size() - 1;
 TB_Point point;
 //右下角
 if(ScoreMtx[rter][cter] > ScoreMtx[rter - 1][cter] &&
    ScoreMtx[rter][cter] > ScoreMtx[rter][cter - 1] &&
    ScoreMtx[rter][cter] > ScoreMtx[rter - 1][cter - 1]) {
  point.Pos0 = rter;
  point.Pos1 = cter;
  point.Score = ScoreMtx[rter][cter];
  TBPoint.push_back(point);
 }
 //右隅
 for(int i = 1; i != rter; ++i) {
  if(ScoreMtx[i][cter] > ScoreMtx[i - 1][cter] &&
     ScoreMtx[i][cter] > ScoreMtx[i][cter - 1] &&
     ScoreMtx[i][cter] > ScoreMtx[i - 1][cter - 1] &&
     ScoreMtx[i][cter] > ScoreMtx[i + 1][cter]) {
   TB_Point tmp(i, cter, ScoreMtx[i][cter]);
   TBPoint.push_back(tmp);
  }
 }
 //下隅
 for(int j = 1; j != cter; ++j) {
  if(ScoreMtx[rter][j] > ScoreMtx[rter - 1][j] &&
     ScoreMtx[rter][j] > ScoreMtx[rter][j - 1] &&
     ScoreMtx[rter][j] > ScoreMtx[rter - 1][j - 1] &&
     ScoreMtx[rter][j] > ScoreMtx[rter][j + 1]) {
   TB_Point tmp(rter, j, ScoreMtx[rter][j]);
   TBPoint.push_back(tmp);
  }
 }

 //その他
 for(int i = 1; i != rter; ++i) {
  for(int j = 1; j != cter; ++j) {
   if(ScoreMtx[i][j] > ScoreMtx[i - 1][j] &&
      ScoreMtx[i][j] > ScoreMtx[i][j - 1] &&
      ScoreMtx[i][j] > ScoreMtx[i - 1][j - 1] &&
      ScoreMtx[i][j] > ScoreMtx[i + 1][j] &&
      ScoreMtx[i][j] > ScoreMtx[i][j + 1] &&
      ScoreMtx[i][j] > ScoreMtx[i + 1][j + 1]) {
    TB_Point tmp(i, j, ScoreMtx[i][j]);
    TBPoint.push_back(tmp);
   }
  }
 }

 //ラムダ式を引数にしたソート
 std::sort(TBPoint.begin(), TBPoint.end(), [](auto const &l, auto const &r) { return ( l.Score > r.Score ); } );

 return &TBPoint;
}

const std::array<std::vector<int>, 2> *
        Dynamic_Programming::traceback(int n)
{
 //TBPoint の n 番目を構築するので，呼び出されるたびに再構築
 for(std::vector<int> &vect : AlignedNum) {
  vect.clear();
  vect.shrink_to_fit();
  }

 if(TBPoint.empty())
  return &AlignedNum;

 TBter0 = TBPoint[n].Pos0;
 TBter1 = TBPoint[n].Pos1;
 //スコアが 0 になるか，どちらかが端まで進んだら終了
 while(ScoreMtx[TBter0][TBter1] > 0.0 && TBter0 > 0 && TBter1 > 0) {
  if(TraceMtx[TBter0][TBter1] == 'X') {
   //scoreMtx は 1 残基分ずれている．以下同
   AlignedNum[0].push_back(--TBter0);
   AlignedNum[1].push_back(--TBter1);
  }
  else if(TraceMtx[TBter0][TBter1] == '-') {
   AlignedNum[0].push_back(-1);
   AlignedNum[1].push_back(--TBter1);
  }
  else {  //if(traceMtx[TBter0][TBter1] == '|')
   AlignedNum[0].push_back(--TBter0);
   AlignedNum[1].push_back(-1);
  }
 }

 //if 内は，'|' などで 1 残基のみ一致した場合（アラインメントではない）の対処
 if(AlignedNum[0].size() == 1 || AlignedNum[0].size() == 1) {
  AlignedNum[0].clear();
  AlignedNum[0].shrink_to_fit();
  AlignedNum[1].clear();
  AlignedNum[1].shrink_to_fit();
 }
 else {
  std::reverse(AlignedNum[0].begin(), AlignedNum[0].end());
  std::reverse(AlignedNum[1].begin(), AlignedNum[1].end());
 }
 return &AlignedNum;
}

double Dynamic_Programming::get_score(int num)
{
 if(num > (int)TBPoint.size())
  return -1.0;
 return TBPoint[num].Score;
}

