#include "file_io.h"
#include <iomanip>
#include <queue>
#include <sstream>

File_IO::File_IO()
 : lineNum(1),
   peptName('A'),
   atomVect(nullptr),
   serialToIdx(nullptr) {}

bool File_IO::read_xyz(const std::string *file_name, std::vector<Atom> *atom_vect, std::string *comment) {
 try {
  std::ifstream fi(*file_name);

  std::string line;
  getline(fi, line);  //1 行めは原子数
  std::stringstream strm(line);
  int atom_cnt(0);
  strm >> atom_cnt;

  atom_vect->assign(atom_cnt, Atom());  //原子数 == サイズ

  getline(fi, *comment);  //2 行めはコメント
  if(comment->empty())
   *comment = "This data was created with Builcule.";

  for(Atom &atom : *atom_vect) {  //3 行め以降は元素と座標）
   getline(fi, line);
   std::stringstream strm(line);
   strm >> atom.Symbol;
   strm >> atom.XYZ[0];
   strm >> atom.XYZ[1];
   strm >> atom.XYZ[2];
   atom.serialNum = SERIAL_NUM;  //ここでシリアル番号を設定
  }
 }
 catch(...) {
  std::cerr << __PRETTY_FUNCTION__ << std::endl << "ファイルの読み取りに失敗しました．" << std::endl;
  return false;
 }

 return true;
}
bool File_IO::read_bcl(const std::string *file_name, std::vector<Atom> *atom_vect, std::string *comment) {
 try {
  std::ifstream fi(*file_name);

  std::string line;
  getline(fi, line);  //1 行めは原子数
  std::stringstream strm(line);
  int atom_cnt(0);
  strm >> atom_cnt;

  atom_vect->assign(atom_cnt, Atom());

  getline(fi, *comment);  //2 行めはコメント
  if(comment->empty())
   *comment = "This data was created with Builcule.";

  //原子の情報を (*atom_vect) と bond_idx に取り込む
  int idx(-1);
  for(int i = 0; i < atom_cnt; ++i) {  //3 行め以降は元素と座標）
   getline(fi, line);
   std::stringstream strm(line);
   Atom &atom = (*atom_vect)[i];
   strm >> atom.Symbol;
   strm >> atom.XYZ[0];
   strm >> atom.XYZ[1];
   strm >> atom.XYZ[2];
   strm >> idx;  //当該原子のインデックス番号は，取り込む必要なし
   while(strm >> idx)
    atom.Bond.push_back(idx); //ここではインデックスが設定される
   atom.serialNum = SERIAL_NUM;  //ここでシリアル番号を設定
  }

  //結合情報をインデックスからシリアルへ変換
  for(int i = 0; i != atom_cnt; ++i) {
   std::vector<int> &bond = (*atom_vect)[i].Bond;
   int bond_cnt = bond.size();
   for(int j = 0; j != bond_cnt; ++j)
    bond[j] = (*atom_vect)[bond[j]].serialNum;
  }
 }
 catch(...) {
  std::cerr << __PRETTY_FUNCTION__ << std::endl << "ファイルの読み取りに失敗しました．" << std::endl;
  return false;
 }

 return true;
}

bool File_IO::read_pdb(const std::string *file_name, std::vector<Atom> *atom_vect) {
 try {
  std::ifstream fi(*file_name);

  std::string line;
  std::string atom_symbol;
  while(getline(fi, line)) {
   if(line.substr(0, 4) == "ATOM" || line.substr(0, 6) == "HETATM") {
    Atom atom;
    atom_symbol = line.substr(76, 2);
    if(atom_symbol[0] == ' ')// {
     atom_symbol.erase(atom_symbol.begin());
//    }
//    else {
//     atom_symbol[1] = (char)tolower(atom_symbol[1]);
//    }
    atom.Symbol = atom_symbol;
    //31-37 文字め : X 座標
    atom.XYZ[0] = atof((line.substr(30, 8)).c_str());
    //39-45 文字め : Y 座標
    atom.XYZ[1] = atof((line.substr(38, 8)).c_str());
    //47-53 文字め : Z 座標
    atom.XYZ[2] = atof((line.substr(46, 8)).c_str());
    atom.serialNum = SERIAL_NUM;  //ここでシリアル番号を設定
    (*atom_vect).push_back(atom);
   }
  }
 }
 catch(...) {
  std::cerr << __PRETTY_FUNCTION__ << std::endl << "ファイルの読み取りに失敗しました．" << std::endl;
  return false;
 }

 return true;
}

bool File_IO::read_cif(
        const std::string *file_name,
        std::vector<Atom> *atom_vect) {
 try {
  std::ifstream fi(*file_name);

  std::string line;
  std::string group_PDB;
  std::string tmp;
  while(getline(fi, line)) {
   std::stringstream strm(line);
   strm >> group_PDB;
   if(group_PDB == "ATOM" || group_PDB == "HETATM") {
    Atom atom;
    strm >> tmp;
    strm >> atom.Symbol;
    for(int i = 0; i < 7; ++i)
     strm >> tmp;
    for(int i = 0; i < 3; ++i)
     strm >> atom.XYZ[i];
    atom.serialNum = SERIAL_NUM;  //ここでシリアル番号を設定
    (*atom_vect).push_back(atom);
   }
  }
 }
 catch(...) {
  std::cerr << __PRETTY_FUNCTION__ << std::endl << "ファイルの読み取りに失敗しました．" << std::endl;
  return false;
 }

 return true;
}

bool File_IO::read_fasta(const std::string *file_name,
        std::vector<std::string> *comment,
        std::vector<std::vector<char>> *sequence)
{
 try {
  std::ifstream fi(*file_name);
  //eof で failbit が設定されるので，badbit のみ検出
  fi.exceptions(std::ios::badbit);
  std::string tmp_line;
  std::string tmp_sequence;

  while(getline(fi, tmp_line)) {
   if(tmp_line.empty())
    break;
   else if(tmp_line[0] == '>') {  //コメント行の場合．新しい配列が始まる
    comment->push_back(tmp_line);
    sequence->push_back(std::vector<char>());
   }
   else {
    std::vector<char> &seq = sequence->back();
    int n = tmp_line.size();
    for(int i = 0; i < n; ++i)
     seq.push_back(tmp_line[i]);
   }
  }
 }
 catch(...) {
  comment->clear();
  comment->shrink_to_fit();
  sequence->clear();
  sequence->shrink_to_fit();
  return false;
 }
 return true;
}

bool File_IO::write_bcl_or_xyz(const std::string *file_name, const std::string * type, const std::vector<Atom> *atom_vect, const std::map<int, int> *serial_to_idx, std::string *comment) {
 std::ofstream fo;
 fo.exceptions(std::ios::failbit | std::ios::badbit);
 try {
  fo.open(*file_name, std::ios::out);
 }
 catch(...) {
  std::cerr << "ファイルのオープンに失敗しました．" << std::endl;
  return false;
 }

 if(comment->empty())
  *comment = "This file was creatwd with Builcule.";

 try {
 int num = 0;
  fo << std::setw(10) << std::left << atom_vect->size() << std::endl;  //原子数
  fo << *comment << std::endl;
  for(const Atom &atom : *atom_vect) {  //元素と座標
   fo << std::setw( 3) << std::left << SYMBOL[atom.Elmt]
      << std::setw(15) << std::right << std::fixed << std::setprecision(5) << atom.XYZ[0]
     << std::setw(15) << std::right << std::fixed << std::setprecision(5) << atom.XYZ[1]
      << std::setw(15) << std::right << std::fixed << std::setprecision(5) << atom.XYZ[2];

   if(*type == "bcl") {
    fo << std::setw(10) << std::right << num++;
    for(int serial : atom.Bond)
     fo << std::setw(10) << std::right << serial_to_idx->at(serial);
    }
   fo << std::endl;
  }
 }
 catch(...) {
  fo.close();
  std::cerr << "ファイルの書き込みに失敗しました．" << std::endl;
  return false;
 }

 try { fo.close(); }
 catch(...) {
  std::cerr << "ファイルのクローズに失敗しました．" << std::endl;
  return false;
 }

 return true;
}

bool File_IO::write_pdb(const std::string *file_name, const std::vector<Mol> *mol_vect, const std::vector<Atom> *atom_vect, const std::map<int, int> *serial_to_idx, std::string *comment) {

 atomVect = atom_vect;
 serialToIdx = serial_to_idx;
 atomCheck.assign(atom_vect->size(), 1);  //チェックしたら 0 にする
 if(comment->empty())
  *comment = "This file was creatwd with Builcule.";
 lineNum = 1;  //行番号．1 行出力したら増加
 peptName = 'A';  //ペプチド名
 AACode = "XXX";

 std::ofstream fo;
 fo.exceptions(std::ios::failbit | std::ios::badbit);
 try {
  fo.open(*file_name, std::ios::out);
 }
 catch(...) {
  std::cerr << "ファイルのオープンに失敗しました．" << std::endl;
  return false;
 }

 try {
  int mol_num = mol_vect->size();
  for(int i = 0; i < mol_num; ++i) {
   int pept_cnt = (*mol_vect)[i].aaTable.size();
   for(int j = 0; j < pept_cnt; ++j) {
    int aa_cnt = (*mol_vect)[i].aaTable[j].size();
    for(int k = 0; k < aa_cnt; ++k) {
     aaNum = k + 1;  //23-26 : アミノ酸番号
     const Amino_Acid &aa = (*mol_vect)[i].aaTable[j][k];
     AACode[0] = aa.Code3[0];
     AACode[1] = toupper(aa.Code3[1]);
     AACode[2] = toupper(aa.Code3[2]);
     write_aa(fo, aa);
    }
    //欠落している場合は OXT == -1 となっているはず
    if((*mol_vect)[i].aaTable[j][aa_cnt -1].OXT > 0)
     write_aa_atm(fo, (*mol_vect)[i].aaTable[j][aa_cnt -1].OXT, "OXT");
    if(aa_cnt)
     write_ter(fo);
    peptName == 'Z' ? 'A' : ++peptName;  //22 : ペプチド名
   }
  }

  write_hetero(fo);
 }
 catch(...) {
  fo.close();
  std::cerr << "ファイルの書き込みに失敗しました．" << std::endl;
  return false;
 }
 try { fo.close(); }
 catch(...) {
  std::cerr << "ファイルのクローズに失敗しました．" << std::endl;
  return false;
 }

 return true;
}

void File_IO::write_aa(std::ofstream &fo, const Amino_Acid &aa) {
 //ペプチド主鎖．OXT は別途出力している
 write_aa_atm(fo, aa.Annot.at(N), "N");
 write_aa_atm(fo, aa.Annot.at(CA), "CA");
 write_aa_atm(fo, aa.Annot.at(C), "C");
 write_aa_atm(fo, aa.Annot.at(O), "O");

 //ペプチド側鎖
 if(aa.Annot.find(CB) != aa.Annot.end())
  write_aa_atm(fo, aa.Annot.at(CB), "CB");
 if(aa.Annot.find(CD) != aa.Annot.end())
  write_aa_atm(fo, aa.Annot.at(CD), "CD");
 if(aa.Annot.find(CD1) != aa.Annot.end())
  write_aa_atm(fo, aa.Annot.at(CD1), "CD1");
 if(aa.Annot.find(CD2) != aa.Annot.end())
  write_aa_atm(fo, aa.Annot.at(CD2), "CD2");
 if(aa.Annot.find(CE) != aa.Annot.end())
  write_aa_atm(fo, aa.Annot.at(CE), "CE");
 if(aa.Annot.find(CE1) != aa.Annot.end())
  write_aa_atm(fo, aa.Annot.at(CE1), "CE1");
 if(aa.Annot.find(CE2) != aa.Annot.end())
  write_aa_atm(fo, aa.Annot.at(CE2), "CE2");
 if(aa.Annot.find(CG) != aa.Annot.end())
  write_aa_atm(fo, aa.Annot.at(CG), "CG");
 if(aa.Annot.find(CG1) != aa.Annot.end())
  write_aa_atm(fo, aa.Annot.at(CG1), "CG1");
 if(aa.Annot.find(CG2) != aa.Annot.end())
  write_aa_atm(fo, aa.Annot.at(CG2), "CG2");
 if(aa.Annot.find(CH2) != aa.Annot.end())
  write_aa_atm(fo, aa.Annot.at(CH2), "CH2");
 if(aa.Annot.find(CZ) != aa.Annot.end())
  write_aa_atm(fo, aa.Annot.at(CZ), "CZ");
 if(aa.Annot.find(CZ2) != aa.Annot.end())
  write_aa_atm(fo, aa.Annot.at(CZ2), "CZ2");
 if(aa.Annot.find(CZ3) != aa.Annot.end())
  write_aa_atm(fo, aa.Annot.at(CZ3), "CZ3");
 if(aa.Annot.find(ND1) != aa.Annot.end())
  write_aa_atm(fo, aa.Annot.at(ND1), "ND1");
 if(aa.Annot.find(ND2) != aa.Annot.end())
  write_aa_atm(fo, aa.Annot.at(ND2), "ND2");
 if(aa.Annot.find(NE) != aa.Annot.end())
  write_aa_atm(fo, aa.Annot.at(NE), "NE");
 if(aa.Annot.find(NE1) != aa.Annot.end())
  write_aa_atm(fo, aa.Annot.at(NE1), "NE1");
 if(aa.Annot.find(NE2) != aa.Annot.end())
  write_aa_atm(fo, aa.Annot.at(NE2), "NE2");
 if(aa.Annot.find(NH1) != aa.Annot.end())
  write_aa_atm(fo, aa.Annot.at(NH1), "NH1");
 if(aa.Annot.find(NH2) != aa.Annot.end())
  write_aa_atm(fo, aa.Annot.at(NH2), "NH2");
 if(aa.Annot.find(NZ) != aa.Annot.end())
  write_aa_atm(fo, aa.Annot.at(NZ), "NZ");
 if(aa.Annot.find(OD) != aa.Annot.end())
  write_aa_atm(fo, aa.Annot.at(OD), "OD");
 if(aa.Annot.find(OD1) != aa.Annot.end())
  write_aa_atm(fo, aa.Annot.at(OD1), "OD1");
 if(aa.Annot.find(OD2) != aa.Annot.end())
  write_aa_atm(fo, aa.Annot.at(OD2), "OD2");
 if(aa.Annot.find(OE1) != aa.Annot.end())
  write_aa_atm(fo, aa.Annot.at(OE1), "OE1");
 if(aa.Annot.find(OE2) != aa.Annot.end())
  write_aa_atm(fo, aa.Annot.at(OE2), "OE2");
 if(aa.Annot.find(OG) != aa.Annot.end())
  write_aa_atm(fo, aa.Annot.at(OG), "OG");
 if(aa.Annot.find(OG1) != aa.Annot.end())
  write_aa_atm(fo, aa.Annot.at(OG1), "OG1");
 if(aa.Annot.find(OH) != aa.Annot.end())
  write_aa_atm(fo, aa.Annot.at(OH), "OH");
 if(aa.Annot.find(SD) != aa.Annot.end())
  write_aa_atm(fo, aa.Annot.at(SD), "SD");
 if(aa.Annot.find(SG) != aa.Annot.end())
  write_aa_atm(fo, aa.Annot.at(SG), "SG");

 //ペプチド H
 for(int serial : aa.mainH)
  write_aa_atm(fo, serial, "H");
 for(int serial : aa.sideH)
  write_aa_atm(fo, serial, "H");
}

void File_IO::write_aa_atm(std::ofstream &fo, int serial, const std::string &elmt) {
 int idx = serialToIdx->at(serial);
 atomCheck[idx] = 0;  //チェック済みにする

 const Eigen::Vector3d &xyz = (*atomVect)[idx].XYZ;
 fo << std::setw(6) << std::left << "ATOM";

 fo << std::setw(5) << std::right << lineNum;  //行番号
 fo << std::setw(2) << ' ';
 fo << std::setw(3) << std::left << elmt;  //元素名
 fo << ' ';
 fo << std::setw(3) << AACode;  //アミノ酸コード
 fo << ' ';
 fo << std::setw(1) << peptName;  //ペプチド名（アルファベット）
 fo << std::setw(4) << std::right << aaNum;  //アミノ酸番号（ペプチドごと）
 fo << std::setw(5) << ' ';
 fo << std::fixed << std::setprecision(3);
 fo << std::setw(7) << xyz[0] << ' ';
 fo << std::setw(7) << xyz[1] << ' ';
 fo << std::setw(7) << xyz[2] << ' ';
 fo << std::setw(23) << elmt[0];  //注番号（元素）
 fo << std::endl;

 ++lineNum;
}

void File_IO::write_ter(std::ofstream &fo) {
 fo << std::setw(6) << std::left << "TER";
 fo << std::setw(5) << std::right << lineNum++;  //行番号
 fo << std::endl;
}

void File_IO::write_hetero(std::ofstream &fo) {
//原子のシリアル番号ではなく， atomVect のインデックスで管理
 int idx = get_hetero_mol_ini();
 while(idx > -1) {  //idx == 0 がヘテロ原子の場合がある
  push_hetero_mol(idx);
  idx = get_hetero_mol_ini();
 }

 //多原子分子
 peptName = 'A';  //分子名として使う
 int mol_cnt = hetMol.size();
 for(int i = 0; i < mol_cnt; ++i) {
  int atom_cnt = hetMol[i].size();
  for(int j = 0; j < atom_cnt; ++j) {
   write_hetero_atom(fo, hetMol[i][j]);
  }
  peptName == 'Z' ? 'A' : ++peptName;
 }

 //単原子分子
 peptName = 'A';
 int atom_cnt = atomCheck.size();
 for(int i = 0; i < atom_cnt; ++i)
  if(atomCheck[i])
   write_hetero_atom(fo, i);

 //ヘテロ原子の結合情報
 write_connect(fo);
}

int File_IO::get_hetero_mol_ini() {
 //ヘテロ分子を hetMol に格納．単原子分子は検知しない
 int atom_cnt = atomCheck.size();
 for(int i = 0; i < atom_cnt; ++i)
  if(atomCheck[i] && !(*atomVect)[i].Bond.empty())
   return i;

 return -1;
}

void File_IO::push_hetero_mol(int idx) {
 std::vector<int> het_mol;
 std::queue<int> tmpq;
 tmpq.push(idx);

 while(!tmpq.empty()) {
  int idx0 = tmpq.front();
  tmpq.pop();
  het_mol.push_back(idx0);
  atomCheck[idx0] = 0;  //(1)
  for(int serial : (*atomVect)[idx0].Bond) {
   int idx1 = serialToIdx->at(serial);
   if(atomCheck[idx1]) {
    tmpq.push(idx1);
    atomCheck[idx1] = 0;  //(1) だけでは環に対応できない
   }
  }
 }

 hetMol.push_back(het_mol);
}

void File_IO::write_hetero_atom(std::ofstream &fo, int idx) {
 const Atom &atom = (*atomVect)[idx];
 serialToLineNum[atom.serialNum] = lineNum;

 std::string elmt = atom.Symbol;
 if(elmt.size() > 1) elmt[1] = toupper(elmt[1]);
 if(elmt.size() > 2) elmt[2] = toupper(elmt[2]);
 if(atom.Bond.empty() && elmt == "O") elmt = "HOH";
 Eigen::Vector3d xyz = atom.XYZ;
 fo << std::setw(6) << std::left << "HETATM";
 fo << std::setw(5) << std::right << lineNum++;
 fo << std::setw(2) << ' ';
 fo << std::setw(3) << std::left << elmt;
 fo << std::setw(5) << ' ';
 fo << std::setw(1) << peptName;  //分子名（アルファベット）
 fo << std::setw(9) << ' ';
 fo << std::fixed << std::setprecision(3) << std::right;
 fo << std::setw(7) << xyz[0] << ' ';
 fo << std::setw(7) << xyz[1] << ' ';
 fo << std::setw(7) << xyz[2] << ' ';
 fo << std::setw(20) << ' ';
 fo << std::setw(3) << std::right << elmt;
 fo << std::endl;
}

void File_IO::write_connect(std::ofstream &fo) {
 for(const std::vector<int> &het_mol : hetMol) {
  if(het_mol.size() > 1) {
   for(int serial0 : het_mol) {
    if(serialToLineNum[serial0] > 0) {
     fo << std::setw(6) << std::left << "CONECT";
     fo << std::setw(5) << std::right << serialToLineNum[serial0];
     int idx = serialToIdx->at(serial0);
     const std::vector<int> &bond = (*atomVect)[idx].Bond;
     for(int serial1 : bond) {
      int line_num = serialToLineNum[serial1];
      if(line_num)
       fo << std::setw(5) << std::right << line_num;
     }
     fo << std::endl;
    }
   }
  }
 }
}

bool File_IO::write_fasta(const std::string *file_name, const std::vector<Mol> *mol_vect) {
 std::ofstream fo;
 fo.exceptions(std::ios::failbit | std::ios::badbit);
 try {
  fo.open(*file_name, std::ios::out);
 }
 catch(...) {
  std::cerr << "ファイルのオープンに失敗しました．" << std::endl;
  return false;
 }

 try {
  int num = 0;
  int mol_cnt = mol_vect->size();
  for(int i = 0; i < mol_cnt; ++i) {
   const std::vector<std::vector<Amino_Acid>> &aa_table = (*mol_vect)[i].aaTable;
   int pept_cnt = aa_table.size();
   for(int j = 0; j < pept_cnt; ++j) {
    fo << ">num : " << num++ << ", mol num : " << i << ", pept num : " << j << std::endl;
    const std::vector<Amino_Acid> &pept = aa_table[j];
    int aa_cnt = pept.size();
    int char_cnt = 1;
    for(int k = 0; k < aa_cnt; ++k) {
     fo << pept[k].Code1;
     if(++char_cnt > 80) {
      fo << std::endl;
      char_cnt = 1;
     }
    }
    fo << std::endl;
   }
  }
 }
 catch(...) {
  fo.close();
  std::cerr << "ファイルの書き込みに失敗しました．" << std::endl;
  return false;
 }

 try { fo.close(); }
 catch(...) {
  std::cerr << "ファイルのクローズに失敗しました．" << std::endl;
  return false;
 }

 return true;
}

void File_IO::export_for_babel_mm(
        const std::vector<Atom> *atom_vect,
        const std::map<int, int> *serial_to_idx) {
 //隠しディレクトリに全原子を HETATM とする PDB 形式で出力
 std::string home(std::getenv("HOME"));
 std::string name_in = home + "/.builcule/babel_in.pdb";

 std::ofstream fo;
 fo.exceptions(std::ios::failbit | std::ios::badbit);
 try {
  fo.open(name_in, std::ios::out);
 }
 catch(...) {
  std::cerr << __PRETTY_FUNCTION__ << std::endl;
  std::cerr << "ファイルのオープンに失敗しました．" << std::endl;
  return;
 }

 try {
  //原子情報
  int cnt = 0;
  for(const Atom &atom : *atom_vect) {
   fo << std::setw(6) << std::left << "HETATM";
   fo << std::setw(5) << std::right << cnt++;
   fo << std::setw(2) << ' ';
   fo << std::setw(3) << std::left << atom.Symbol;
   fo << std::setw(5) << ' ';
   fo << std::setw(1) << 'A';  //分子の識別記号は 1 個でいい
   fo << std::setw(9) << ' ';
   fo << std::fixed << std::setprecision(3) << std::right;
   fo << std::setw(7) << atom.XYZ[0] << ' ';
   fo << std::setw(7) << atom.XYZ[1] << ' ';
   fo << std::setw(7) << atom.XYZ[2] << ' ';
   fo << std::setw(20) << ' ';
   fo << std::setw(3) << std::right << atom.Symbol;
   fo << std::endl;
  }

  //結合情報
  cnt = 0;
  for(const Atom &atom : *atom_vect) {
   fo << std::setw(6) << std::left << "CONECT";
   fo << std::setw(5) << std::right << cnt++;
   const std::vector<int> &bond = atom.Bond;
   for(int pair_serial : bond) {
    int pair_idx = serial_to_idx->at(pair_serial);
    fo << std::setw(5) << std::right << pair_idx;
   }
   fo << std::endl;
  }
 }
 catch(...) {
  std::cerr << __PRETTY_FUNCTION__ << std::endl;
  std::cerr << "ファイルの書き込みに失敗しました．" << std::endl;
  fo.close();
  return;
 }
 try { fo.close(); }
 catch(...) {
  std::cerr << __PRETTY_FUNCTION__ << std::endl;
  std::cerr << "ファイルのクローズに失敗しました．" << std::endl;
 }
}

void File_IO::export_for_babel_mm(
        const std::vector<Atom> *atom_vect,
        const std::map<int, int> *serial_to_idx,
        const std::vector<int> *mol_serial) {
 //隠しディレクトリに全原子を HETATM とする PDB 形式で出力

 //面倒なのでデータ構造を作成
 std::vector<Atom> mol_atom;
 std::map<int, int> mol_serial_map;
 int atom_cnt = mol_serial->size();
 for(int i = 0; i < atom_cnt; ++i) {
  int atom_serial = (*mol_serial)[i];
  int atom_idx = serial_to_idx->at(atom_serial);
  Atom atom = (*atom_vect)[atom_idx];

  int char_cnt = atom.Symbol.size();
  for(int i = 1; i < char_cnt; ++i)
   atom.Symbol[i] = toupper(atom.Symbol[i]);

  mol_atom.push_back(atom);

  mol_serial_map[atom_serial] = i;
 }

 std::string home(std::getenv("HOME"));
 std::string name_in = home + "/.builcule/babel_in.pdb";

 std::ofstream fo;
 fo.exceptions(std::ios::failbit | std::ios::badbit);
 try {
  fo.open(name_in, std::ios::out);
 }
 catch(...) {
  std::cerr << __PRETTY_FUNCTION__ << std::endl;
  std::cerr << "ファイルのオープンに失敗しました．" << std::endl;
  return;
 }

 try {
  int cnt = mol_atom.size();

 //原子情報
  for(int i = 0; i < cnt; ++i) {
   Atom &atom = mol_atom[i];
   fo << std::setw(6) << std::left << "HETATM";
   fo << std::setw(5) << std::right << i;
   fo << std::setw(2) << ' ';
   fo << std::setw(3) << std::left << atom.Symbol;
   fo << std::setw(5) << ' ';
   fo << std::setw(1) << 'A';  //分子の識別記号は 1 個でいい
   fo << std::setw(9) << ' ';
   fo << std::fixed << std::setprecision(3) << std::right;
   fo << std::setw(7) << atom.XYZ[0] << ' ';
   fo << std::setw(7) << atom.XYZ[1] << ' ';
   fo << std::setw(7) << atom.XYZ[2] << ' ';
   fo << std::setw(20) << ' ';
   fo << std::setw(3) << std::right << atom.Symbol;
   fo << std::endl;
  }

 //結合情報
  for(int i = 0; i < cnt; ++i) {
   fo << std::setw(6) << std::left << "CONECT";
   fo << std::setw(5) << std::right << i;
   const std::vector<int> &bond = mol_atom[i].Bond;
   for(int pair_serial : bond) {
    int pair_idx = mol_serial_map[pair_serial];
    fo << std::setw(5) << std::right << pair_idx;
   }
   fo << std::endl;
  }
 }

 catch(...) {
  std::cerr << __PRETTY_FUNCTION__ << std::endl;
  std::cerr << "ファイルの書き込みに失敗しました．" << std::endl;
  fo.close();
  return;
 }
 try { fo.close(); }
 catch(...) {
  std::cerr << __PRETTY_FUNCTION__ << std::endl;
  std::cerr << "ファイルのクローズに失敗しました．" << std::endl;
 }
}

bool File_IO::write_alignment(
        const std::string *file_name,
        const std::vector<std::string> *summary,
        const std::array<std::vector<char>, 2> *alignment) {
 std::ofstream fo;
 fo.exceptions(std::ios::failbit | std::ios::badbit);
 try {
  fo.open(*file_name, std::ios::out);
 }
 catch(...) {
  std::cerr << "ファイルのオープンに失敗しました．" << std::endl;
  return false;
 }

 int char_cnt = (*alignment)[0].size();  //全文字数
 int col_cnt = 70;  //1 行 70 文字とする
 int times = char_cnt / col_cnt + 1;
 int ini = 0;
 int ter = col_cnt;
 if(ter >= char_cnt)
  ter = char_cnt;
 try {
  //要旨
  for(std::string str : *summary)
   fo << str << std::endl;
  fo << std::endl;

  //アラインメント
  for(int i = 0; i < times; ++i) {
   for(int j = ini; j < ter; ++j)
    fo << (*alignment)[0][j];
   fo << std::endl;

   for(int j = ini; j < ter; ++j) {
    if(toupper((*alignment)[0][j]) != toupper((*alignment)[1][j]))
     fo << '_';
    else
     fo << '*';
   }
   fo << std::endl;

   for(int j = ini; j < ter; ++j)
    fo << (*alignment)[1][j];
   fo << std::endl;
  }

  fo << std::endl;

  ini += col_cnt;
  ter += col_cnt;
  if(ter >= char_cnt)
   ter = char_cnt;
 }
 catch(...) {
  fo.close();
  std::cerr << "ファイルの書き込みに失敗しました．" << std::endl;
  return false;
 }

 try { fo.close(); }
 catch(...) {
  std::cerr << "ファイルのクローズに失敗しました．" << std::endl;
  return false;
 }

 return true;
}

bool File_IO::read_1_col_data(
        const std::string *file_name,
        std::vector<Atom> *atom_vect) {
 std::vector<double> charge;
 double val(0.0);
 try {
  std::ifstream fi(*file_name);
  std::string line;
  while(1) {
   getline(fi, line);
   if(line.empty()) break;
   std::stringstream strm(line);
   strm >> val;
   charge.push_back(val);
  }
 }
 catch(...) {
  std::cerr << __PRETTY_FUNCTION__ << std::endl << "ファイルの読み取りに失敗しました．" << std::endl;
  return false;
 }

 int atom_cnt = atom_vect->size();
 if(atom_cnt != (int)charge.size()) {
  std::cerr << __PRETTY_FUNCTION__ << std::endl << "原子数と電荷数が不一致です．" << std::endl;
  return false;
 }
 for(int i = 0; i < atom_cnt; ++i)
  (*atom_vect)[i].Charge = charge[i];
 return true;
}

