#include "mol.h"
#include <algorithm>

Mol::Mol(std::vector<Atom> *atom_vect, const std::map<int, int> *map)
 : atomVect(atom_vect),
   serialToIdx(map),
   adjAtom(atomVect, serialToIdx)
{
 if(!atomVect || !serialToIdx) {
  std::cerr << __PRETTY_FUNCTION__ << std::endl;
  exit(1);
 }
}

bool Mol::is_bound(int serial0, int serial1) {
 int idx0 = serialToIdx->at(serial0);
 const std::vector<int> &bond0 = (*atomVect)[idx0].Bond;
 int idx1 = serialToIdx->at(serial1);
 const std::vector<int> &bond1 = (*atomVect)[idx1].Bond;
 bool b0 = find(bond0.begin(), bond0.end(), serial1) == bond0.end();
 bool b1 = find(bond1.begin(), bond1.end(), serial0) == bond1.end();
 if(b0 != b1) {
  std::cerr << __PRETTY_FUNCTION__ << std::endl;
  std::cerr << serial0 << " と " << serial1 << " の間の結合の状態が不正です．";
 }
 return !(b0 && b1);
}

void Mol::detect_main_chain() {
 aaTable.clear();
 aaTable.shrink_to_fit();
 detectedAA.clear();
 detectedAA.shrink_to_fit();
 if(atomSerial.size() < 8)  //Gly-Gly を検知の下限とする
  return;

 //アミノ酸の検知
 scan_C();

 for(Amino_Acid &aa : detectedAA) {
  scan_CA(aa);
  if(aa.AA)
   scan_N(aa);
 }

 //Ser 側鎖による誤検知を削除
 int aa_cnt = detectedAA.size();
 for(int i = 0; i < aa_cnt; ++i) {
  for(int j = 0; j < i; ++j)
   if(detectedAA[i].Annot[CA] == detectedAA[j].Annot[CA])
    clear_ser_duplication(i, j);
  for(int j = i + 1; j < aa_cnt; ++j)
   if(detectedAA[i].Annot[CA] == detectedAA[j].Annot[CA])
    clear_ser_duplication(i, j);
 }
}

void Mol::scan_C() {
 Amino_Acid aa;
 for(int serial : atomSerial) {
  const Atom &atom = (*atomVect)[serialToIdx->at(serial)];

  if(atom.Elmt == 6) {
   adjAtom.check(serial);

   if(!adjAtom.S.empty() || !adjAtom.X.empty());  //C でない
   else if(adjAtom.C.size() == 1 &&  //一般的な C
      adjAtom.N.size() == 1 &&
      adjAtom.O.size() == 1) {
    aa.AA = true;
    aa.Annot[C] = serial;
    aa.Annot[CA] = adjAtom.C[0];
    aa.Annot[O] = adjAtom.O[0];
    aa.nextN = adjAtom.N[0];
   }
   else if(adjAtom.C.size() == 1 && adjAtom.O.size() == 2) {  //C-末
    aa.AA = true;
    aa.Annot[C] = serial;
    aa.Annot[CA] = adjAtom.C[0];
    aa.Annot[O] = adjAtom.O[0];
    if(adjAtom.O[1] > 0)
     aa.OXT = adjAtom.O[1];
   }
   else if(adjAtom.C.size() == 1 && adjAtom.O.size() == 1) {  //Ser かも
    aa.AA = true;
    aa.Annot[C] = serial;
    aa.Annot[CA] = adjAtom.C[0];
    aa.Annot[O] = adjAtom.O[0];
   }
  }

  if(aa.Annot[C] > 0) {
   aa.AA = true;
   detectedAA.push_back(aa);
  }
  aa.init();
 }
}

void Mol::scan_CA(Amino_Acid &aa) {
 adjAtom.check(aa.Annot[CA]);
 if(!adjAtom.O.empty() || !adjAtom.S.empty() || !adjAtom.X.empty()) {
  aa.init();
  return;
 }

 if(adjAtom.N.size() != 1) {
  aa.init();
  return;
 }
 int c_cnt = adjAtom.C.size();
 if(c_cnt != 1 && c_cnt != 2) {  //G の場合は 1 個
  aa.init();
  return;
 }

 aa.Annot[N] = adjAtom.N[0];

 if(adjAtom.C.size() == 2)
  aa.Annot[CB]
      = (adjAtom.C[0] == aa.Annot[C]) ? adjAtom.C[1] : adjAtom.C[0];
 else {
  aa.confirm_g();
 }

 if(!adjAtom.H.empty())
  aa.mainH.insert(aa.mainH.end(), adjAtom.H.begin(), adjAtom.H.end());
}

void Mol::scan_N(Amino_Acid &aa) {
 //水素はキャプチャする（修飾残基はキャプチャしない）
 int serialN = aa.Annot[N];
 adjAtom.check(serialN);
 if(!adjAtom.H.empty())
  aa.mainH.insert(aa.mainH.end(), adjAtom.H.begin(), adjAtom.H.end());

 for(const Amino_Acid &tmp_aa : detectedAA) {
  if(tmp_aa.AA && tmp_aa.nextN == serialN) {
  aa.terN = false;
  return;
  }
 }

 aa.terN = true;
 //Pro の場合は，側鎖のキャプチャ時に判断
}

void Mol::clear_ser_duplication(int aa0, int aa1) {
 if(!detectedAA[aa0].nextN && detectedAA[aa1].nextN)
  detectedAA[aa0].init();
 else if(detectedAA[aa0].nextN && !detectedAA[aa1].nextN)
  detectedAA[aa1].init();
 else if(!detectedAA[aa0].nextN && !detectedAA[aa1].nextN)
  detectedAA[aa0].init();  //どちらでもよい
 else if(detectedAA[aa0].nextN && detectedAA[aa1].nextN) {
  detectedAA[aa0].init();  //ペプチドが分岐しているので両方初期化
  detectedAA[aa1].init();
 }
}

int Mol::search_next_aa_pos(int next_n) {
 int aa_cnt = detectedAA.size();
 for(int i = 0; i != aa_cnt; ++i) {
  if(detectedAA[i].Annot[N] == next_n && detectedAA[i].AA) {
   return i;
  }
 }

 return -1;
}

bool Mol::is_modifyed_n_ter(int aa_pos) {
 int aa_n = detectedAA[aa_pos].Annot[N];
 for(const Amino_Acid &aa : detectedAA) {
  if(aa_n == aa.nextN)
   return false;
 }
 detectedAA[aa_pos].terN = true;
 return true;
}

void Mol::sequencing() {
 int aa_cnt = detectedAA.size();
 //N-末が遊離アミノ基の場合
 for(int i = 0; i != aa_cnt; ++i)
  if(detectedAA[i].terN && !detectedAA[i].Peptide)
   sequencing(i);

 //N-末が修飾されている場合
 for(int i = 0; i != aa_cnt; ++i) {
  if(!detectedAA[i].Peptide &&
      detectedAA[i].AA &&
      detectedAA[i].nextN &&
      is_modifyed_n_ter(i)) {
   sequencing(i);
  }
 }
}

void Mol::sequencing(int aa_pos) {
 std::vector<Amino_Acid> tmp_pept;

 detectedAA[aa_pos].Peptide = true;
 tmp_pept.push_back(detectedAA[aa_pos]);

 while(tmp_pept.back().nextN) {
  aa_pos = search_next_aa_pos(tmp_pept.back().nextN);
  if(aa_pos < 0)
   break;
  detectedAA[aa_pos].Peptide = true;
  tmp_pept.push_back(detectedAA[aa_pos]);
 }

 if(tmp_pept.size() > 1)
  aaTable.push_back(tmp_pept);
}

void Mol::capture_side_chain() {
 for(std::vector<Amino_Acid> &peptide : aaTable) {
  for(Amino_Acid &aa : peptide) {
   if(aa.AA) {  //アミノ酸として認識されていれば
    aa.confirm_ca_cb();
    if(aa.Code1 != 'G')
     capture_side_chain(aa);  //側鎖からアミノ酸を同定
    if(aa.Code1 != 'X')  //通常アミノ酸
     aa.erase_dupl_side_h();  //水素は重複してキャプチャされている
    else  //異常アミノ酸
     aa.confirm_x();
    aa.push_all_serial();
   }
   else {  //アミノ酸でないならば
    aa.init();
   }
  }
 }
}

void Mol::capture_side_chain(Amino_Acid &aa) {
//lv0 は CA，lv1 は CB

 capture_lv(aa, 2);
 const std::array<std::vector<int>, 6> &lv2 = aa.sideChainC[2];
 int c2 = lv2[1].size();
 int o2 = lv2[3].size();
 int s2 = lv2[4].size();
 if(!lv2[2].empty() || !lv2[5].empty()) {
  aa.confirm_x();
  return;
 }
 else if(c2 > 3 || o2 > 1 || s2 > 1) {
  aa.confirm_x();
  return;
 }
 else if(!c2 && !o2 && !s2) {  //側鎖が CB のみ
  aa.confirm_a();
  return;
 }

 capture_lv(aa, 3);
 const std::array<std::vector<int>, 6> &lv3 = aa.sideChainC[3];
 if(!lv3[5].empty()) {
  aa.confirm_x();
  return;
 }
 int c3 = lv3[1].size();
 int n3 = lv3[2].size();
 int o3 = lv3[3].size();
 int s3 = lv3[4].size();
 if(c3 > 2 || n3 > 1 || o3 > 2 || s3 > 1) {
  aa.confirm_x();
  return;
 }
 else if(!c3 && !n3 && !o3 && !s3) {
  if(c2 == 2 && !o2 && !s2) {
   aa.confirm_v();
   return;
  }
  else if(c2 == 1 && o2 == 1 && !s2) {
   aa.confirm_t();
   return;
  }
  else if(!c2 && o2 == 1 && !s2) {
   aa.confirm_s();
   return;
  }
  else if(!c2 && !o2 && s2 == 1) {
   aa.confirm_C();
   return;
  }
 }
 else if(!c3 && !n3 && !o3 && s3 == 1) {  //S-S 結合
  if(!c2 && !o2 && s2 == 1) {
   aa.confirm_c();
   return;
  }
 }

 capture_lv(aa, 4);
 const std::array<std::vector<int>, 6> &lv4 = aa.sideChainC[4];
 if(!lv4[4].empty() || !lv4[5].empty()) {
  aa.confirm_x();
  return;
 }

 int c4 = lv4[1].size();
 int n4 = lv4[2].size();
 int o4 = lv4[3].size();
 if(c4 > 2 || n4 > 1 || o4 > 2) {
  aa.confirm_x();
  return;
 }
 else if(!c4 && !n4 && !o4) {
  if(c2 == 1 && !o2 && !s2 &&
     c3 == 2 && !n3 && !o3 && !s3) {
   aa.confirm_l();
   return;
  }
  else if(c2 == 1 && !o2 && !s2 &&
          !c3 && n3 == 1 && o3 == 1 && !s3) {
   aa.confirm_n();
   return;
  }
  else if(c2 == 1 && !o2 && !s2 &&
          !c3 && !n3 && o3 == 2 && !s3) {
   aa.confirm_d();
   return;
  }
  else if(c2 == 2 && !o2 && !s2 &&
          c3 == 1 && !n3 && !o3 && !s3) {
   //is_bound() を使うのでここで同定
   aa.Code1 = 'I';  aa.Code3 ="Ile";
   if(is_bound(lv2[1][0], lv3[1][0])) {
    aa.Annot[CG1] = lv2[1][0];
    aa.Annot[CG2] = lv2[1][1];
   }
   else {
    aa.Annot[CG1] = lv2[1][1];
    aa.Annot[CG2] = lv2[1][0];
   }
   aa.Annot[CD1] = lv3[1][0];
   aa.confirm_i();
   return;
  }
 }
 else if(!c4 && n4 == 1 && !o4) {
 //is_bound() と o3 を使うのでここで同定
  if(c2 == 1 && !o2 && !s2 &&
     c3 == 1 && !n3 && o3 < 2 && !s3 &&
     lv4[2][0] == aa.Annot[N] &&
     is_bound(lv3[1][0], aa.Annot[N])) {
   aa.Code1 = 'P';  aa.Code3 ="Pro";
   aa.Annot[CG] = lv2[1][0];
   aa.Annot[CD] = lv3[1][0];
   if(o3) {
    aa.Code1 = 'p';  aa.Code3 ="Hpy";
    aa.Annot[OD] = lv3[3][0];
   }
   adjAtom.check(aa.Annot[N]);
   int c_cnt = adjAtom.C.size();
   if(c_cnt == 2) {
    aa.terN = true;
   }
   aa.confirm_p();
   return;
  }
 }

 capture_lv(aa, 5);
 const std::array<std::vector<int>, 6> &lv5 = aa.sideChainC[5];
 if(!lv5[3].empty() || !lv5[4].empty() || !lv5[5].empty()) {
  aa.confirm_x();
  return;
 }

 int c5 = lv5[1].size();
 int n5 = lv5[2].size();
 if(c5 > 3 || n5 > 1) {  //環では多重カウントされている
  aa.confirm_x();
  return;
 }
 else if(!c5 && !n5) {
  if(c2 == 1 && !o2 && !s2 &&
     !c3 && !n3 && !o3 && s3 == 1 &&
     c4 == 1 && !n4 && !o4) {
   aa.confirm_m();
   return;
  }
  else if(c2 == 1 && !o2 && !s2 &&
          c3 == 1 && !n3 && !o3 && !s3 &&
          !c4 && n4 == 1 && o4 == 1) {
   aa.confirm_q();
   return;
  }
  else if(c2 == 1 && !o2 && !s2 &&
          c3 == 1 && !n3 && !o3 && !s3 &&
          !c4 && !n4 && o4 == 2) {
   aa.confirm_e();
   return;
  }
 }
 else if(c2 == 1 && !o2 && !s2 &&
         c3 == 1 && n3 == 1 && !o3 && !s3 &&
         c4 == 1 && n4 == 1 && !o4 &&
         c5 == 1 && n5 == 1) {
  if(lv4[1][0] == lv5[1][0] && lv4[2][0] == lv5[2][0]) {
   aa.confirm_h();
   return;
  }
 }

 capture_lv(aa, 6);
 const std::array<std::vector<int>, 6> &lv6 = aa.sideChainC[6];
 if(!lv6[4].empty() || !lv6[5].empty()) {
  aa.confirm_x();
  return;
 }

 int c6 = lv6[1].size();
 int n6 = lv6[2].size();
 int o6 = lv6[3].size();
 if(c6 > 5 || n6 > 2 || o6 > 2) {  //環では多重カウントされている
  aa.confirm_x();
  return;
 }
 else if(!c6 && !n6 && !o6 &&
     c2 == 1 && !o2 && !s2 &&
     c3 == 1 && !n3 && !o3 && !s3 &&
     c4 == 1 && !n4 && !o4 &&
     !c5 && n5 == 1) {
   aa.confirm_k();
  return;
 }
 else if(c6 == 0 && !n6 && !o6 &&
         c2 == 1 && !o2 && !s2 &&
         c3 == 2 && !n3 && !o3 && !s3 &&
         c4 == 2 && !n4 && !o4 &&
         c5 == 2 && !n5) {
   aa.confirm_f();
 }

 capture_lv(aa, 7);
 const std::array<std::vector<int>, 6> &lv7 = aa.sideChainC[7];
 if(!lv7[2].empty() || !lv7[3].empty() || !lv7[4].empty() || !lv7[5].empty()) {
  aa.confirm_x();
  return;
 }

 int c7 = lv7[1].size();
 if(c7 != 4 && c7 != 0) {
  aa.confirm_x();
  return;
 }
 if(!c7) {
  if(c2 == 1 && !o2 && !s2 &&
     c3 == 1 && !n3 && !o3 && !s3 &&
     !c4 && n4 == 1 && !o4 &&
     c5 == 1 && !n5 &&
     !c6 && n6 == 2 && !o6) {
   aa.confirm_r();
   return;
  }
  else if(c2 == 1 && !o2 && !s2 &&
          c3 == 2 && !n3 && !o3 && !s3 &&
          c4 == 2 && !n4 && !o4 &&
          c5 == 2 && !n5 &&
          !c6 && !n6 && o6 == 2) {
   if(lv5[1][0] == lv5[1][1] && lv6[3][0] == lv6[3][1]) {
   aa.confirm_y();
   return;
   }
  }
 }
 else if(c2 == 1 && !o2 && !s2 &&
         c3 == 2 && !n3 && !o3 && !s3 &&
         c4 == 2 && n4 == 1 && !o4 &&
         c5 == 3 && n5 == 1 &&
         c6 == 5 && !n6 && !o6){
 //adjAtom を使うのでここで同定
  aa.Annot[CG] = lv2[1][0];

  adjAtom.check(lv3[1][0]);
  if(adjAtom.N.empty()) {
   aa.Annot[CD1] = lv3[1][1];
   aa.Annot[CD2] = lv3[1][0];
  }
  else {
   aa.Annot[CD1] = lv3[1][0];
   aa.Annot[CD2] = lv3[1][1];
  }

  aa.Annot[NE1] = lv4[2][0];
  if(is_bound(aa.Annot[NE1], lv4[1][0])) {
   aa.Annot[CE1] = lv4[1][0];
   aa.Annot[CE2] = lv4[1][1];
  }
  else {
   aa.Annot[CE1] = lv4[1][1];
   aa.Annot[CE2] = lv4[1][0];
  }

  adjAtom.check(aa.Annot[CE1]);
  if(adjAtom.C[0] == aa.Annot[CD2])
   aa.Annot[CZ2] = adjAtom.C[1];
  else
   aa.Annot[CZ2] = adjAtom.C[0];

  adjAtom.check(aa.Annot[CE2]);
  if(adjAtom.C[0] == aa.Annot[CD2])
   aa.Annot[CZ3] = adjAtom.C[1];
  else
   aa.Annot[CZ3] = adjAtom.C[0];

  adjAtom.check(aa.Annot[CZ2]);
  if(adjAtom.C[0] == aa.Annot[CE1])
   aa.Annot[CH2] = adjAtom.C[1];
  else
   aa.Annot[CH2] = adjAtom.C[0];

  aa.Code1 = 'W';  aa.Code3 ="Trp";
  aa.confirm_w();
 }
}

void Mol::capture_lv(Amino_Acid &aa, int lv) {
 //側鎖の原子を sideChainC[lv] に取り込んで，sideChainR[lv] に再構成
 //同一原子が複数回取り込まれる．水素は，後で一括して整理
 //lv１- 1 層の原子を root にして lv 層の原子を取り込む
 std::vector<int> &leaf_r = aa.sideChainR[lv];
 for(int serial : aa.sideChainR[lv - 1]) {
  adjAtom.check(serial);
  aa.sideH.insert(aa.sideH.end(), adjAtom.H.begin(), adjAtom.H.end());
  leaf_r.insert(leaf_r.end(), adjAtom.C.begin(), adjAtom.C.end());
  leaf_r.insert(leaf_r.end(), adjAtom.N.begin(), adjAtom.N.end());
  leaf_r.insert(leaf_r.end(), adjAtom.O.begin(), adjAtom.O.end());
  leaf_r.insert(leaf_r.end(), adjAtom.S.begin(), adjAtom.S.end());
  leaf_r.insert(leaf_r.end(), adjAtom.X.begin(), adjAtom.X.end());
 }

 //2 層下の原子も取り込まれているので削除
 const std::vector<int> &ref = aa.sideChainR[lv - 2];
 for(int serial : ref)
  leaf_r.erase(remove(leaf_r.begin(), leaf_r.end(), serial), leaf_r.end());

 //sideChainR[lv] を sideChainC[lv] に再構成
 std::array<std::vector<int>, 6> &leaf_c = aa.sideChainC[lv];
 for(int serial : aa.sideChainR[lv]) {
  const Atom &atom = (*atomVect)[serialToIdx->at(serial)];
  if(atom.Elmt == 6)
   leaf_c[1].push_back(serial);
  else if(atom.Elmt == 7)
   leaf_c[2].push_back(serial);
  else if(atom.Elmt == 8)
   leaf_c[3].push_back(serial);
  else if(atom.Elmt == 16)
   leaf_c[4].push_back(serial);
  else if(atom.Elmt != 1)
   leaf_c[5].push_back(serial);
 }
}

const std::vector<char> *Mol::get_aa_seq(int pept_num) {
 peptSeq.clear();
 peptSeq.shrink_to_fit();

 int aa_cnt = aaTable[pept_num].size();
 peptSeq.assign(aa_cnt, 'X');
 for(int i = 0; i != aa_cnt; ++i)
  peptSeq[i] = aaTable[pept_num][i].Code1;

 return &peptSeq;
}

void Mol::update_protein_num_data() {
 int pept_cnt = aaTable.size();
 if(!pept_cnt)
  return;

 for(int i = 0; i < pept_cnt; ++i) {
  int aa_cnt = aaTable[i].size();
  for(int j = 0; j < aa_cnt; ++j) {
   for(int serial : aaTable[i][j].allSerial) {
    int idx = serialToIdx->at(serial);
    (*atomVect)[idx].peptIdx = i;
    (*atomVect)[idx].aaIdx = j;
    (*atomVect)[idx].aaCode = aaTable[i][j].Code1;
   }
  }
 }
}

